package com.instabug.library.screenshot.instacapture

import android.app.Activity
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Paint
import android.graphics.Rect
import android.widget.Toast
import com.instabug.library.Constants
import com.instabug.library.R
import com.instabug.library.core.InstabugCore
import com.instabug.library.instacapture.ActivityReferenceManager
import com.instabug.library.internal.utils.memory.IBGLowMemroyWarning
import com.instabug.library.screenshot.ScreenshotCaptor
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.LocaleUtils
import com.instabug.library.util.extenstions.constructErrorMessage
import com.instabug.library.util.memory.MemoryUtils
import java.util.concurrent.Future

class ScreenshotRequestArgs(
    @ScreenShotType val screenShotType: Int,
    val activity: Activity,
    val listener: ScreenshotCaptor.CapturingCallback
)

interface ScreenshotRequest {
    val listener: ScreenshotCaptor.CapturingCallback
    val activity: ActivityReferenceManager
    fun start()

    companion object Factory {
        @JvmStatic
        fun createScreenshotRequest(args: ScreenshotRequestArgs): ScreenshotRequest {
            val hierarchyExtractor = HierarchyExtractor.create()
            val activityReferenceManager = ActivityReferenceManager(args.activity)
            val capturingStrategy = CapturingStrategy.fromScreenshotType(args.screenShotType)
            return ScreenshotRequestImpl(
                listener = args.listener,
                activity = activityReferenceManager,
                capturingStrategy = capturingStrategy,
                hierarchyExtractor = hierarchyExtractor
            )
        }

    }
}

class ScreenshotRequestImpl(
    override val listener: ScreenshotCaptor.CapturingCallback,
    override val activity: ActivityReferenceManager,
    private val capturingStrategy: CapturingStrategy,
    private val hierarchyExtractor: HierarchyExtractor,
) : ScreenshotRequest {
    override fun start() {
        val activity = activity.validatedActivity
        if (activity == null) {
            listener.onCapturingFailure(IBGActivityNotRunningException("Can't capture screenshot due to null activity"))
            return
        }
        if (isMemoryLow(activity, listener)) return
        InstabugSDKLogger.d(Constants.LOG_TAG, "start capture screenshot")
        captureAndMask(activity)
    }

    private fun isMemoryLow(
        currentActivity: Activity?,
        screenshotCapturingListener: ScreenshotCaptor.CapturingCallback
    ): Boolean {
        return if (MemoryUtils.isLowMemory(currentActivity)) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG, "Couldn't take initial screenshot due to low memory"
            )
            screenshotCapturingListener.onCapturingFailure(IBGLowMemroyWarning("Your activity is currently in low memory"))
            val message = LocaleUtils.getLocaleStringResource(
                InstabugCore.getLocale(currentActivity),
                R.string.instabug_str_capturing_screenshot_error,
                currentActivity
            )
            Toast.makeText(currentActivity, message, Toast.LENGTH_SHORT).show()
            true
        } else false
    }

    private fun captureAndMask(activity: Activity) {
        val maskingRects = hierarchyExtractor.extract(activity)
        capturingStrategy.captureScreenshot(
            activity, screenCaptureListener(maskingRects)
        )
    }

    private fun screenCaptureListener(maskingRects: Future<List<Rect>>) =
        object : ScreenshotCaptor.CapturingCallback by listener {
            override fun onCapturingSuccess(bitmap: Bitmap) {
                runCatching { bitmap.mask(maskingRects.get()) }
                    .onSuccess(listener::onCapturingSuccess)
                    .onFailure { exception ->
                        val message = constructErrorMessage("couldn't mask bitmap", exception)
                        InstabugSDKLogger.w(Constants.LOG_TAG, message)
                        IBGMaskingException(
                            "Couldn't mask screenshot",
                            exception
                        ).also(listener::onCapturingFailure)
                    }
            }
        }

    private fun Bitmap.mask(maskingRects: List<Rect>): Bitmap {
        Canvas(this)
            .apply {
                val paint = Paint()
                maskingRects.forEach { drawRect(it, paint) }
            }
        return this
    }
}