package com.instabug.library.screenshot.analytics

import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.diagnostics.nonfatals.model.NonFatal
import com.instabug.library.screenshot.analytics.ibgthrowables.IBGUnKnownException
import java.util.concurrent.atomic.AtomicInteger

class ScreenshotAnalytics(
    private val _screenShotCaptureCount: AtomicInteger = AtomicInteger(),
    private val _screenShotDroppedCount: AtomicInteger = AtomicInteger(),
    private val _capturedErrorCodes: MutableMap<String, AtomicInteger> = mutableMapOf()
) {

    val screenShotCaptureCount: Int get() = _screenShotCaptureCount.get()
    val screenShotDroppedCount: Int get() = _screenShotDroppedCount.get()
    val capturedErrorCodes: Map<String, AtomicInteger> get() = _capturedErrorCodes

    fun onScreenShotSuccessfullyCaptured() {
        _screenShotCaptureCount.incrementAndGet()
    }

    fun onScreenShotDropped() {
        _screenShotDroppedCount.incrementAndGet()
    }

    fun onScreenshotCaptureError(error: AnalyticsEvent.ErrorEvent) {
        if (error.reportNonFatal) {
            val throwable = error.cause ?: IBGUnKnownException(
                "An error happened due to unknown exception",
                null
            )
            IBGDiagnostics.reportNonFatal(
                throwable,
                "Unknown exception happened", NonFatal.Priority.NORMAL
            )
        }
        if (!_capturedErrorCodes.containsKey(error.errorCode)) {
            _capturedErrorCodes[error.errorCode] = AtomicInteger()
        }
        _capturedErrorCodes[error.errorCode]?.incrementAndGet()
    }

    fun copy(): ScreenshotAnalytics {
        val newErrorCodes =
            _capturedErrorCodes.mapValues { AtomicInteger(it.value.get()) }.toMutableMap()
        return ScreenshotAnalytics(
            AtomicInteger(_screenShotCaptureCount.get()),
            AtomicInteger(_screenShotDroppedCount.get()),
            newErrorCodes
        )
    }

    fun adjustErrorCodes(productsErrorCodes: MutableMap<String, AtomicInteger>) {
        _capturedErrorCodes.putAll(productsErrorCodes)
    }

}

