package com.instabug.library.networkv2.detectors

import android.annotation.SuppressLint
import android.content.Context
import android.content.IntentFilter
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.NetworkRequest
import android.os.Build
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.Constants
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider

object IBGNetworkAvailabilityManager {

    @VisibleForTesting
    var connectivityManager: ConnectivityManager? = null

    @VisibleForTesting
    var isNetworkAvailable = false

    @VisibleForTesting
    val availableNetworks = mutableSetOf<Network>()

    private var areNetworkCallbacksRegistered = false

    @VisibleForTesting
    val requiredNetworkCapabilities: Set<Int> by lazy {
        mutableSetOf<Int>().apply {
            if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP)
                add(NetworkCapabilities.NET_CAPABILITY_INTERNET)
            if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.M)
                add(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
        }
    }

    @VisibleForTesting
    val networkCallbacks: ConnectivityManager.NetworkCallback by lazy {
        @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
        object : ConnectivityManager.NetworkCallback() {

            override fun onAvailable(network: Network) {
                super.onAvailable(network)
                InstabugSDKLogger.v(Constants.LOG_TAG, "network connection available")
                onNetworkAvailable(network)
            }

            override fun onCapabilitiesChanged(
                network: Network, networkCapabilities: NetworkCapabilities
            ) {
                super.onCapabilitiesChanged(network, networkCapabilities)
                handleNetworkCapabilities(network)
            }

            override fun onLost(network: Network) {
                super.onLost(network)
                InstabugSDKLogger.v(Constants.LOG_TAG, "network connection lost")
                onNetworkUnavailable(network)
            }
        }
    }

    private val networkReceiver by lazy { IBGNetworkReceiver() }

    @SuppressLint("NewApi")
    @JvmStatic
    fun registerNetworkCallbacks(context: Context?) {
        context?.let { ctx ->
            connectivityManager = getConnectivityManager(context)
            if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP) {
                if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.M) {
                    connectivityManager?.activeNetwork?.let {
                        handleNetworkCapabilities(it)
                    }
                }
                registerConnectivityManagerCallbacks()
            } else {
                registerNetworkReceiver(ctx)
            }
        }
    }

    @JvmStatic
    @VisibleForTesting
    fun registerNetworkReceiver(context: Context) {
        if (!networkReceiver.isRegistered) {
            val filter = IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION)
            networkReceiver.register(context, filter)
        }
    }

    @JvmStatic
    @VisibleForTesting
    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun registerConnectivityManagerCallbacks() {
        if (!areNetworkCallbacksRegistered) {
            val networkRequest = NetworkRequest.Builder().apply {
                addTransportType(NetworkCapabilities.TRANSPORT_WIFI)
                addTransportType(NetworkCapabilities.TRANSPORT_CELLULAR)
                requiredNetworkCapabilities.forEach { addCapability(it) }
            }.build()

            connectivityManager?.let {
                it.registerNetworkCallback(networkRequest, networkCallbacks)
                areNetworkCallbacksRegistered = true
            }
        }
    }

    @SuppressLint("NewApi")
    @JvmStatic
    fun unregisterNetworkCallbacks(context: Context?) {
        context?.let {
            if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP) {
                unRegisterConnectivityManagerCallbacks()
            } else {
                unregisterNetworkReceiver(it)
            }
        }
    }

    @JvmStatic
    @VisibleForTesting
    fun unregisterNetworkReceiver(context: Context) {
        if (networkReceiver.isRegistered) {
            networkReceiver.unregister(context)
        }
    }

    @JvmStatic
    @VisibleForTesting
    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun unRegisterConnectivityManagerCallbacks() {
        if (areNetworkCallbacksRegistered) {
            connectivityManager?.run {
                unregisterNetworkCallback(networkCallbacks)
                areNetworkCallbacksRegistered = false
            }
        }
    }

    @JvmStatic
    @VisibleForTesting
    fun onNetworkAvailable(network: Network) {
        if (!isNetworkAvailable) {
            //Network was not available last time.
            PoolProvider.postIOTask(object : Runnable {
                //Note: Using lambda expression unexpectedly making the unit test fail!
                override fun run() {
                    IBGCoreEventPublisher.post(IBGSdkCoreEvent.NetworkActivated)
                }
            })
        }
        availableNetworks.add(network)
        isNetworkAvailable = true
    }

    @JvmStatic
    @VisibleForTesting
    fun onNetworkUnavailable(network: Network) {
        if (availableNetworks.contains(network)) {
            availableNetworks.remove(network)
        }
        if (availableNetworks.isEmpty()) {
            isNetworkAvailable = false
        }
    }

    @JvmStatic
    @VisibleForTesting
    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun handleNetworkCapabilities(network: Network) {
        if (isItMeetTheRequiredCapabilities(network)) {
            onNetworkAvailable(network)
        } else {
            onNetworkUnavailable(network)
        }
    }

    private fun getConnectivityManager(context: Context?): ConnectivityManager? {
        try {
            context?.applicationContext?.let { appContext ->
                return (appContext.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager)
            }
        } catch (e: SecurityException) {
            InstabugSDKLogger.w(
                Constants.LOG_TAG,
                """
            Could not read network state. To enable please add the following line in your AndroidManifest.xml <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE"/>
            ${e.message}
            """.trimIndent()
            )
        } catch (e: Exception) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG, "Something went wrong while checking network state", e
            )
        }
        return null
    }

    @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
    fun isItMeetTheRequiredCapabilities(network: Network): Boolean {
        val currentNetworkCapabilities =
            connectivityManager?.getNetworkCapabilities(network)
        return requiredNetworkCapabilities.all {
            currentNetworkCapabilities?.hasCapability(it) ?: false
        }
    }

    fun isOnline(): Boolean {
        return if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP) {
            isNetworkAvailable
        } else {
            isOnlineLegacy()
        }
    }

    @VisibleForTesting
    @JvmStatic
    fun isOnlineLegacy(): Boolean {
        connectivityManager?.activeNetworkInfo?.let { info ->
            return info.isConnectedOrConnecting
        }
        return false
    }
}