package com.instabug.library.networkv2.connection;

import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.util.Map;

/**
 * InstabugBaseConnectionManager is abstraction layer for different connections provided by the {@link com.instabug.library.networkv2.NetworkManager}
 *
 * @see NormalConnectionManager
 * @see MultipartConnectionManager
 * @see FileDownloadConnectionManager
 * @see com.instabug.library.networkv2.request.RequestType
 */
public interface InstabugBaseConnectionManager {

    /**
     Prepare the connection object for the passed {@link Request}

     @return the http connection after being build
     */
    HttpURLConnection create(Request request) throws Exception;

    /**
     Make connection object set and ready according to the connection type with required configurations
     @param connection  the originally built object

     @return the connection object after been modified
     @see NormalConnectionManager
     @see MultipartConnectionManager
     @see FileDownloadConnectionManager
     */
    HttpURLConnection setupConnection(HttpURLConnection connection, Request request) throws Exception;

    /**
     Get the required content type based on Instabug connection type

     @return the content type as String
     */
    String getContentType();

    /**
     Returns setting for read timeout.

     @return an {@code int} that indicates the read timeout value in milliseconds,
     0 return implies that the option is disabled (i.e., timeout of infinity).
     */
    int getReadTimeOut();

    /**
     Returns setting for connect timeout

     @return an {@code int} that indicates the connect timeout value in milliseconds
     */
    int getConnectTimeOut();

    /**
     Handle the request response and close the opened connection

     @return an {@link RequestResponse} by reading the connection stream
     @param connection
     @param request
     */
    RequestResponse handleResponse(HttpURLConnection connection, Request request) throws IOException;

    /**
     Build connection url, a method to prepare the base connection
     it adds {@link Header#CONTENT_TYPE},
     {@link Header#ACCEPT_CHARSET},
     {@link Header#CONTENT_ENCODING},
     {@link Header#AUTHORIZATION}

     {@link Header#AUTHORIZATION} only added if the request is instabug request
     and not get feature request
     @return connection  the connection object

     @throws Exception if any exception encountered.
     */
    HttpURLConnection buildBaseConnection(Request request) throws Exception;

    /**
     Read the connection stream

     @return response as string read from the connection stream
     */
    String convertStreamToString(InputStream is);

    /**
     Read the connection headers

     @return Map of headers keys and values
     */
    Map<String, String> getHeaderFields(HttpURLConnection connection);

    /**
     Handle Server errors if happened
     @param connection  the connection object

     @see RequestResponse.HttpStatusCode._4xx
     @return error response body
     */
    Throwable handleServerError(HttpURLConnection connection)throws Exception;
}
