package com.instabug.library.internal.storage.cache;

import static android.provider.BaseColumns._ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_ATTACHMENT_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_ENCRYPTED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_LOCALE_PATH;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_REPORT_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_URL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.COLUMN_VIDEO_ENCODED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.AttachmentEntry.TABLE_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager;
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg;
import com.instabug.library.model.Attachment;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.List;

public class AttachmentsDbHelper {

    private static final String TAG = "AttachmentsDbHelper";

    //projection
    private final static String[] ATTACHMENT_COLUMN = {
            InstabugDbContract.AttachmentEntry.TABLE_NAME + "." + InstabugDbContract.AttachmentEntry._ID,
            COLUMN_NAME,
            COLUMN_LOCALE_PATH,
            COLUMN_URL,
            COLUMN_TYPE,
            COLUMN_ATTACHMENT_STATE,
            COLUMN_VIDEO_ENCODED,
            COLUMN_DURATION,
            COLUMN_REPORT_ID,
            COLUMN_ENCRYPTED
    };

    /**
     * Inserting an attachment in the database
     *
     * @param attachment instance of {@link Attachment}
     * @return the row ID of the newly inserted row, or -1 if an error occurred
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static synchronized long insert(Attachment attachment, String reportId) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            if (attachment.getId() != -1) {
                values.put(_ID, attachment.getId());
            }
            values.put(COLUMN_URL, attachment.getUrl());
            Attachment.AttachmentState attachmentState = attachment.getAttachmentState();
            if (attachmentState != null) {
                values.put(COLUMN_ATTACHMENT_STATE, attachmentState.name());
            }
            values.put(COLUMN_DURATION, attachment.getDuration());
            values.put(COLUMN_LOCALE_PATH, attachment.getLocalPath());
            values.put(COLUMN_NAME, attachment.getName());
            if (attachment.getType() != null) {
                values.put(COLUMN_TYPE, attachment.getType().name());
            }
            values.put(COLUMN_VIDEO_ENCODED, attachment.isVideoEncoded());
            values.put(COLUMN_REPORT_ID, reportId);
            values.put(COLUMN_ENCRYPTED, attachment.isEncrypted());
            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insertWithOnConflictReplace(TABLE_NAME, null, values);
            db.setTransactionSuccessful();
            return rowId;
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error inserting attachments: " + e.getMessage());
            IBGDiagnostics.reportNonFatal(e, "Error inserting attachments: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
        return -1;
    }

    /**
     * Retrieve a list of attachments related to specific report
     *
     * @param reportId the id of the report
     * @param db       instance of {@link SQLiteDatabaseWrapper}
     * @return a list of {@link Attachment}
     */
    public static ArrayList<Attachment> retrieve(String reportId, SQLiteDatabaseWrapper db) {

        //projection
        String[] ATTACHMENT_COLUMN = {
                InstabugDbContract.AttachmentEntry.TABLE_NAME + "." + InstabugDbContract.AttachmentEntry._ID,
                COLUMN_NAME,
                COLUMN_LOCALE_PATH,
                COLUMN_URL,
                COLUMN_TYPE,
                COLUMN_ATTACHMENT_STATE,
                COLUMN_VIDEO_ENCODED,
                COLUMN_DURATION,
                COLUMN_REPORT_ID,
                COLUMN_ENCRYPTED
        };

        String[] args = {reportId};

        Cursor c = null;
        ArrayList<Attachment> attachments = new ArrayList<>();
        try {
            c = db.query(
                    TABLE_NAME,
                    ATTACHMENT_COLUMN,
                    COLUMN_REPORT_ID + "=?",
                    args,
                    null,
                    null,
                    null,
                    null);
            if (c != null && c.moveToFirst()) {
                do {
                    Attachment attachment = new Attachment();

                    int idColIndex = c.getColumnIndex(_ID);
                    attachment.setId(c.getInt(idColIndex));

                    int nameColIndex = c.getColumnIndex(COLUMN_NAME);
                    attachment.setName(c.getString(nameColIndex));

                    int attachmentStateColIndex = c.getColumnIndex(COLUMN_ATTACHMENT_STATE);
                    attachment.setAttachmentState(Enum.valueOf(Attachment.AttachmentState.class, c.getString(attachmentStateColIndex)));

                    int durationColIndex = c.getColumnIndex(COLUMN_DURATION);
                    attachment.setDuration(c.getString(durationColIndex));

                    int urlColIndex = c.getColumnIndex(COLUMN_URL);
                    attachment.setUrl(c.getString(urlColIndex));

                    int localPathColIndex = c.getColumnIndex(COLUMN_LOCALE_PATH);
                    attachment.setLocalPath(c.getString(localPathColIndex));

                    int isVideoEncodedColIndex = c.getColumnIndex(COLUMN_VIDEO_ENCODED);
                    attachment.setVideoEncoded(c.getInt(isVideoEncodedColIndex) != 0);

                    int typeColIndex = c.getColumnIndex(COLUMN_TYPE);
                    attachment.setType(Enum.valueOf(Attachment.Type.class, c.getString(typeColIndex)));

                    int isEncrypted = c.getColumnIndex(COLUMN_ENCRYPTED);
                    attachment.setEncrypted(c.getInt(isEncrypted) != 0);

                    attachments.add(attachment);
                } while (c.moveToNext());
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while retrieving attachments: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while retrieving attachments: " + e.getMessage());
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return attachments;
    }

    /**
     * Retrieve a list of attachments related to specific report in DB Version 2 (Encrypted DB)
     *
     * @param reportId the id of the report stored in DB Version 2
     * @param db       instance of DB Version 2 manager {@link IBGDbManager}
     * @return a list of {@link Attachment} from the DB version 1
     */
    public static ArrayList<Attachment> retrieve(String reportId, IBGDbManager db) {

        //projection
        String[] ATTACHMENT_COLUMN = {
                InstabugDbContract.AttachmentEntry.TABLE_NAME + "." + InstabugDbContract.AttachmentEntry._ID,
                COLUMN_NAME,
                COLUMN_LOCALE_PATH,
                COLUMN_URL,
                COLUMN_TYPE,
                COLUMN_ATTACHMENT_STATE,
                COLUMN_VIDEO_ENCODED,
                COLUMN_DURATION,
                COLUMN_REPORT_ID,
                COLUMN_ENCRYPTED
        };

        List<IBGWhereArg> args = new ArrayList<>();
        args.add(new IBGWhereArg(reportId, true));

        Cursor c = null;
        ArrayList<Attachment> attachments = new ArrayList<>();
        try {
            c = db.query(
                    TABLE_NAME,
                    ATTACHMENT_COLUMN,
                    COLUMN_REPORT_ID + "=?",
                    args,
                    null,
                    null,
                    null,
                    null);
            if (c != null && c.moveToFirst()) {
                do {
                    Attachment attachment = new Attachment();

                    int idColIndex = c.getColumnIndex(_ID);
                    attachment.setId(c.getInt(idColIndex));

                    int nameColIndex = c.getColumnIndex(COLUMN_NAME);
                    attachment.setName(c.getString(nameColIndex));

                    int attachmentStateColIndex = c.getColumnIndex(COLUMN_ATTACHMENT_STATE);
                    attachment.setAttachmentState(Enum.valueOf(Attachment.AttachmentState.class, c.getString(attachmentStateColIndex)));

                    int durationColIndex = c.getColumnIndex(COLUMN_DURATION);
                    attachment.setDuration(c.getString(durationColIndex));

                    int urlColIndex = c.getColumnIndex(COLUMN_URL);
                    attachment.setUrl(c.getString(urlColIndex));

                    int localPathColIndex = c.getColumnIndex(COLUMN_LOCALE_PATH);
                    attachment.setLocalPath(c.getString(localPathColIndex));

                    int isVideoEncodedColIndex = c.getColumnIndex(COLUMN_VIDEO_ENCODED);
                    attachment.setVideoEncoded(c.getInt(isVideoEncodedColIndex) != 0);

                    int typeColIndex = c.getColumnIndex(COLUMN_TYPE);
                    attachment.setType(Enum.valueOf(Attachment.Type.class, c.getString(typeColIndex)));

                    int isEncrypted = c.getColumnIndex(COLUMN_ENCRYPTED);
                    attachment.setEncrypted(c.getInt(isEncrypted) != 0);

                    attachments.add(attachment);
                } while (c.moveToNext());
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while retrieving attachments: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while retrieving attachments: " + e.getMessage());
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return attachments;
    }

    /**
     * Retrieve a list of attachments related to specific report
     *
     * @param reportId the id of the report
     * @return a list of {@link Attachment}
     */
    public static ArrayList<Attachment> retrieve(String reportId) {
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        ArrayList<Attachment> attachments = new ArrayList<>();
        Cursor c = null;
        try {
            db.beginTransaction();
            String[] args = {reportId};
            c = db.query(
                    TABLE_NAME,
                    ATTACHMENT_COLUMN,
                    COLUMN_REPORT_ID + "=?",
                    args,
                    null,
                    null,
                    null,
                    null);
            if (c != null && c.moveToFirst()) {
                do {
                    Attachment attachment = new Attachment();

                    int idColIndex = c.getColumnIndex(_ID);
                    attachment.setId(c.getInt(idColIndex));

                    int nameColIndex = c.getColumnIndex(COLUMN_NAME);
                    attachment.setName(c.getString(nameColIndex));

                    int attachmentStateColIndex = c.getColumnIndex(COLUMN_ATTACHMENT_STATE);
                    attachment.setAttachmentState(Enum.valueOf(Attachment.AttachmentState.class, c.getString(attachmentStateColIndex)));

                    int durationColIndex = c.getColumnIndex(COLUMN_DURATION);
                    attachment.setDuration(c.getString(durationColIndex));

                    int urlColIndex = c.getColumnIndex(COLUMN_URL);
                    attachment.setUrl(c.getString(urlColIndex));

                    int localPathColIndex = c.getColumnIndex(COLUMN_LOCALE_PATH);
                    attachment.setLocalPath(c.getString(localPathColIndex));

                    int isVideoEncodedColIndex = c.getColumnIndex(COLUMN_VIDEO_ENCODED);
                    attachment.setVideoEncoded(c.getInt(isVideoEncodedColIndex) != 0);

                    int typeColIndex = c.getColumnIndex(COLUMN_TYPE);
                    attachment.setType(Enum.valueOf(Attachment.Type.class, c.getString(typeColIndex)));

                    int isEncrypted = c.getColumnIndex(COLUMN_ENCRYPTED);
                    attachment.setEncrypted(c.getInt(isEncrypted) > 0);

                    attachments.add(attachment);
                } while (c.moveToNext());
            }
            db.setTransactionSuccessful();
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while retrieving attachments: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while retrieving attachments: " + e.getMessage());
        } finally {
            if (c != null) {
                c.close();
            }
            db.endTransaction();
            db.close();
        }
        return attachments;
    }

    /**
     * Update a specific attachment that is already in the database
     *
     * @param id the id of the attachment
     * @param cv {@link ContentValues} params of the values that needs to be updated
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static synchronized void update(long id, ContentValues cv) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = _ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(id)};
        db.beginTransaction();
        try {
            db.update(TABLE_NAME, cv, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while updating attachment: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while updating attachment: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Update a specific attachment that is already in the database
     *
     * @param name     the name of the attachment
     * @param reportId the id of the report
     * @param cv       {@link ContentValues} params of the values that needs to be updated
     */
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static synchronized void update(String name, String reportId, ContentValues cv) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_NAME + "=? and " + COLUMN_REPORT_ID + "=? ";
        String[] whereArgs = new String[]{name, reportId};
        db.beginTransaction();
        try {
            db.update(TABLE_NAME, cv, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete a specific attachment record
     *
     * @param id the id of the attachment
     */
    public static synchronized void delete(long id) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = _ID + "=? ";
        String[] whereArgs = new String[]{String.valueOf(id)};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while deleting attachment: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while updating attachment: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete a specific attachment record
     *
     * @param name     the name of the attachment
     * @param reportId the id of the report
     */
    public static synchronized void delete(String name, String reportId) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_NAME + " = ? and " + COLUMN_REPORT_ID + " = ?";
        String[] whereArgs = new String[]{name, reportId};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while deleting attachment: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while deleting attachment: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * Delete all attachments stored in the database
     * i.e. deleting the whole table
     */
    public static synchronized void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } catch (Exception e) {
            InstabugSDKLogger.e(TAG, "Error while deleting all attachments: " + e.getMessage(), e);
            IBGDiagnostics.reportNonFatal(e, "Error while deleting all attachments: " + e.getMessage());
        } finally {
            db.endTransaction();
            db.close();
        }
    }
}
