package com.instabug.library.internal.dataretention.files;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.internal.dataretention.core.Contract;
import com.instabug.library.internal.dataretention.core.DataInspector;
import com.instabug.library.internal.dataretention.core.Disposable;
import com.instabug.library.internal.dataretention.core.PolicyCollector;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

public class FileInspector extends DataInspector<FileDisposalPolicy> {

    @VisibleForTesting
    FileInspector() {
        super(new PolicyCollector<FileDisposalPolicy>() {
            @NonNull
            @Override
            public Collection<FileDisposalPolicy> collect() {
                return Collections.emptyList();
            }
        });
    }

    public FileInspector(@NonNull FilePolicyCollector collector) {
        super(collector);
    }

    public static FileInspector create() {
        return new FileInspector(new FilePolicyCollector());
    }

    @NonNull
    @Override
    public Collection<Disposable> getDisposables() {
        Collection<Disposable> disposables = new LinkedList<>();
        for (FileDisposalPolicy policy : collector.collect()) {
            disposables.addAll(getDisposables(policy));
        }
        return disposables;
    }

    @NonNull
    @VisibleForTesting
    Collection<Disposable> getDisposables(FileDisposalPolicy policy) {
        return getDisposables(policy.scope(), policy.retentionContract());
    }

    /**
     * This is where all the evaluations happen. The following checks are done:
     * Record by record, it checks whether the max retention period defined by
     * {@link Contract#maxRetentionPeriodMillis()} has been exceed or not. It does so by calling
     * {@link #getExpiredRecords(Collection, long)}
     *
     * @param scope          the scope of data to be evaluated
     * @param contract       the contract where all the constraints are defined (i.e. max allowed size)
     * @return a list of all the disposable records
     * @see #getDisposables()
     * @see #getDisposables(FileDisposalPolicy)
     * @see #sortDesc(Collection)
     * @see #getExpiredRecords(Collection, long)
     * @see #toDisposables(Collection)
     */
    @NonNull
    @VisibleForTesting
    Collection<Disposable> getDisposables(FileScope scope, Contract contract) {
        Collection<FileRecord> records = scope.records();
        if (contract.shouldPurge()) return toDisposables(records);
        long maxRetentionPeriod = contract.maxRetentionPeriodMillis();
        Set<FileRecord> disposableRecords = new HashSet<>(getExpiredRecords(records, maxRetentionPeriod));
        return toDisposables(disposableRecords);
    }

    /**
     * Record by record, it checks whether the max retention period defined by
     * {@link Contract#maxRetentionPeriodMillis()} has been exceed or not. If yes, it then checks
     * whether or not this record has any exception rules. If not it adds it to the disposables list
     *
     * @param records            the list of all records
     * @param maxRetentionPeriod the max record age
     * @return a list of all the disposable records
     */
    @NonNull
    @VisibleForTesting
    Collection<FileRecord> getExpiredRecords(Collection<FileRecord> records,
                                             long maxRetentionPeriod) {
        Collection<FileRecord> expiredRecords = new LinkedList<>();
        for (FileRecord record : records) {
            if (record.getAge() > maxRetentionPeriod ) {
                expiredRecords.add(record);
            }
        }
        return expiredRecords;
    }

    /**
     * Sorts the collection of records from least recent to most recent according to the record age
     *
     * @param records the list of all records
     * @return a list of sorted records
     * @see FileRecord#getAge()
     */
    @NonNull
    List<FileRecord> sortDesc(Collection<FileRecord> records) {
        List<FileRecord> recordList = records instanceof List
                ? (List<FileRecord>) records
                : new ArrayList<>(records);
        Collections.sort(recordList, new Comparator<FileRecord>() {
            @Override
            public int compare(FileRecord o1, FileRecord o2) {
                return Double.compare(o2.getAge(), o1.getAge());
            }
        });
        return recordList;
    }

    /**
     * Maps the collection of records to a collection of disposables
     *
     * @param records the list of all records
     * @return a list of disposables
     * @see FileDisposable
     */
    @NonNull
    Collection<Disposable> toDisposables(Collection<FileRecord> records) {
        Collection<Disposable> disposables = new LinkedList<>();
        for (FileRecord record : records) {
            disposables.add(new FileDisposable(record));
        }
        return disposables;
    }
}
