package com.instabug.library.internal.dataretention.core;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

/**
 * Represents a disposable resource.
 */
public interface Disposable {

    /**
     * Dispose the resource, the operation should be idempotent.
     */
    void dispose();

    /**
     * A disposable container that can hold onto multiple other disposables
     */
    class CompositeDisposable implements Disposable {

        @VisibleForTesting
        public final Collection<Disposable> disposables;

        public CompositeDisposable(@NonNull Disposable... disposables) {
            this(new HashSet<>(Arrays.asList(disposables)));
        }

        public CompositeDisposable(@NonNull Collection<Disposable> disposables) {
            this.disposables = disposables;
        }

        public void add(@NonNull Disposable disposable) {
            disposables.add(disposable);
        }

        /**
         * Dispose the contents of the OpenHashSet
         */
        @Override
        public void dispose() {
            for (Disposable disposable : disposables) {
                disposable.dispose();
            }
        }
    }
}
