package com.instabug.library.featuresflags.di

import androidx.annotation.VisibleForTesting
import com.instabug.library.featuresflags.caching.FeaturesFlagDBManager
import com.instabug.library.featuresflags.caching.FeaturesFlagDBManagerImpl
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsHandler
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsHandlerImpl
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsProvider
import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsProviderImpl
import com.instabug.library.featuresflags.managers.FeatureFlagsManager
import com.instabug.library.featuresflags.managers.FeatureFlagsManagerImpl
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.util.threading.PoolProvider
import java.lang.ref.WeakReference


object FeaturesFlagServiceLocator {
    private val featureFlagObjectMap = mutableMapOf<String, WeakReference<Any?>?>()

    @JvmStatic
    val featuresFlagsConfigsHandler: FeatureFlagsConfigsHandler
        @Synchronized get() =
            FeatureFlagsConfigsHandler::class.toString().let { key ->
                (getIfKeyRefExist(key)?.let { it as FeatureFlagsConfigsHandler }
                    ?: run {
                        FeatureFlagsConfigsHandlerImpl(
                            featuresFlagsConfigsProvider,
                            featureFlagsManager
                        ).also { featuresFlagsConfigsHandler ->
                            saveRefValues(key, featuresFlagsConfigsHandler)

                        }
                    })
            }

    @JvmStatic
    val featuresFlagsConfigsProvider: FeatureFlagsConfigsProvider
        @Synchronized get() =
            FeatureFlagsConfigsProvider::class.toString().let { key ->
                (getIfKeyRefExist(key)?.let { it as FeatureFlagsConfigsProvider }
                    ?: run {
                        FeatureFlagsConfigsProviderImpl().also { featuresFlagsConfigsProvider ->
                            saveRefValues(key, featuresFlagsConfigsProvider)

                        }
                    })
            }

    @JvmStatic
    val featuresFlagDBManager: FeaturesFlagDBManager
        @Synchronized get() = FeaturesFlagDBManager::class.toString().let { key ->
            getIfKeyRefExist(key)?.let { available -> available as FeaturesFlagDBManagerImpl }
                ?: run {
                    val dbManager = try { IBGDbManager.getInstance() } catch (e: Exception) { null}
                    FeaturesFlagDBManagerImpl(dbManager).also { featuresFlagDBManagerImplObj ->
                        saveRefValues(key, featuresFlagDBManagerImplObj)

                    }
                }
        }
    @JvmStatic
    val featureFlagsManager: FeatureFlagsManager
        @Synchronized get() =
            FeatureFlagsManager::class.toString().let { key ->
                (getIfKeyRefExist(key)?.let { it as FeatureFlagsManager }
                    ?: run {
                        FeatureFlagsManagerImpl(
                            featuresFlagsConfigsProvider,
                            featuresFlagDBManager,
                            PoolProvider.getInstance().orderedExecutor
                        ).also { featuresFlagsManager ->
                            saveRefValues(key, featuresFlagsManager)

                        }
                    })
            }


    //helper methods
    private fun getIfKeyRefExist(key: String): Any? = featureFlagObjectMap[key]?.get()
    private fun saveRefValues(key: String, value: Any) {
        featureFlagObjectMap[key] = WeakReference(value)
    }

    @VisibleForTesting
    fun reset() {
        featureFlagObjectMap.clear()
    }
}