package com.instabug.library.diagnostics.sdkEvents

import com.instabug.library.Constants
import com.instabug.library.apichecker.APIChecker
import com.instabug.library.diagnostics.isCalledInternally
import com.instabug.library.diagnostics.sdkEvents.configurations.SDKEventConfigs
import com.instabug.library.diagnostics.sdkEvents.di.SDKEventsServiceLocator
import com.instabug.library.diagnostics.sdkEvents.models.SDKEvent
import com.instabug.library.util.InstabugSDKLogger

class SDKEventsHandler {

    private val configs: SDKEventConfigs
        get() = SDKEventsServiceLocator.sdkEventConfigs
    private val sdkEventsManager: SDKEventsManager
        get() = SDKEventsServiceLocator.sdkEventsManager

    fun logEvent(key: String, count: Int, stackTrace: Array<StackTraceElement?>?) {
        APIChecker.checkAndRunInExecutor("SDKEvents.logSDKEvent") {
            val event = SDKEvent(key, count)
            handleEventLoggingLogic(stackTrace, "logEvent()", event) {
                sdkEventsManager.saveEvent(event)
            }
        }
    }

    fun logEventImmediately(key: String, count: Int, stackTrace: Array<StackTraceElement?>?) {
        val event = SDKEvent(key, count)
        handleEventLoggingLogic(stackTrace, "logEventImmediately()", event) {
            sdkEventsManager.saveEventImmediately(event)
        }
    }

    private fun handleEventLoggingLogic(
        stackTrace: Array<StackTraceElement?>?,
        methodName: String,
        event: SDKEvent,
        callBack: () -> Unit
    ) = when {
        isCalledInternally(stackTrace).not() -> logError(methodName)
        configs.isEnabled.not() -> logFeatureNotEnabled()
        configs.allowList?.contains(event.key)?.not() ?: true -> logEventNotAllowed(event.key)
        else -> {
            logSavingEvent(event)
            callBack()
        }
    }

    private fun logError(methodName: String) = InstabugSDKLogger.e(
        Constants.LOG_TAG,
        "Please refrain from using IBGDiagnostics.$methodName() as it is a private API"
    )

    private fun logFeatureNotEnabled() = InstabugSDKLogger.d(
        Constants.LOG_TAG,
        "SDKEvent recording is DISABLED"
    )

    private fun logEventNotAllowed(eventKey: String) = InstabugSDKLogger.e(
        Constants.LOG_TAG,
        "recording event with key: $eventKey is not allowed"
    )

    private fun logSavingEvent(event: SDKEvent) =
        InstabugSDKLogger.v(Constants.LOG_TAG, "saving sdkEvent: $event")
}