package com.instabug.library.diagnostics.nonfatals;


import static com.instabug.library.diagnostics.DiagnosticsUtilsKt.isAnInternalException;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator;
import com.instabug.library.diagnostics.nonfatals.model.NonFatal;
import com.instabug.library.settings.PerSessionSettings;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;

/**
 * Created by Shannan on 06/09/2021.
 */
public class NonFatals {


    private static final String INSTABUG_PREFIX = "com.instabug.";
    private static final String METHOD_REPORT_ERROR = "reportError";
    private static final String METHOD_REPORT_NON_FATAL = "reportNonFatal";

    public static void reportNonFatal(@NonNull final Throwable throwable, @NonNull final String message, @NonFatal.Priority int priority) {

        if (!isAnInternalException(throwable.getStackTrace())) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Please refrain from using NonFatals.reportNonFatal as it is a private API");
            return;
        }

        if (!canReportNonFatal()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "NonFatals disabled temporarily");
            return;
        }

        StackTraceElement instabugElement = getFirstInstabugElementFromStacktrace(throwable.getStackTrace());

        APIChecker.checkAndRunInExecutor("NonFatals.reportNonFatal", () -> {
            if (InstabugCore.getFeatureState(IBGFeature.NON_FATAL_ERRORS) == Feature.State.DISABLED) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "NonFatals reporting is DISABLED");
                return;
            }
            if (instabugElement != null) {
                NonFatalsRecursionDetector.onNonFatalReported(instabugElement.getFileName());
            }
            createFormattedException(throwable, instabugElement, message, priority);
        });
    }

    public static void reportNonFatal(@NonNull final Throwable throwable, @NonNull final String message) {
        reportNonFatal(throwable, message, NonFatal.Priority.NORMAL);
    }

    public static Future<Boolean> reportNonFatalWithPromise(@NonNull final Throwable throwable, @NonNull final String message, @NonFatal.Priority int priority) {
        FutureTask<Boolean> negativeFuture = new FutureTask<>(() -> false);
        negativeFuture.run();
        if (!isAnInternalException(throwable.getStackTrace())) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Please refrain from using NonFatals.reportNonFatal as it is a private API");
            return negativeFuture;
        }

        if (!canReportNonFatal()) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "NonFatals disabled temporarily");
            return negativeFuture;
        }

        StackTraceElement instabugElement = getFirstInstabugElementFromStacktrace(throwable.getStackTrace());

        if (InstabugCore.getFeatureState(IBGFeature.NON_FATAL_ERRORS) == Feature.State.DISABLED) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "NonFatals reporting is DISABLED");
            return negativeFuture;
        }
        if (instabugElement != null) {
            NonFatalsRecursionDetector.onNonFatalReported(instabugElement.getFileName());
        }
        return createFormattedException(throwable, instabugElement, message, priority);
    }

    public static void reportNonFatalAndLog(@NonNull final Throwable throwable, @NonNull final String message, @NonNull String logTag) {
        reportNonFatalAndLog(throwable, message, NonFatal.Priority.NORMAL, logTag);
    }

    public static void reportNonFatalAndLog(@NonNull final Throwable throwable, @NonNull final String message, @NonFatal.Priority int priority, @NonNull String logTag) {
        reportNonFatalWithPromise(throwable, message, priority);
        InstabugSDKLogger.e(logTag, message);
    }


    private static Future<Boolean> createFormattedException(@NonNull Throwable throwable, @Nullable StackTraceElement stackTraceElements, @NonNull String message, @NonFatal.Priority int priority) {
        Future<Boolean> returnable;
        try {
            InstabugSDKLogger.v(Constants.LOG_TAG, "parsing nonfatal: " +
                    throwable.getClass().getCanonicalName());
            NonFatal nonFatal = NonFatalFormatter.createNonFatal(throwable, stackTraceElements, message, priority);
            returnable = reportException(nonFatal);
        } catch (Exception e) {
            FutureTask<Boolean> negativeFuture = new FutureTask<>(()->false);
            negativeFuture.run();
            returnable = negativeFuture;
            InstabugSDKLogger.e(Constants.LOG_TAG, "parsing nonfatal error ", e);
        }
        return returnable;
    }

    private static Future<Boolean> reportException(final NonFatal nonFatal) {
        return ServiceLocator.getNonFatalsThreadExecutor().submit(() -> {
            NonFatalsManager nonFatalsManager = ServiceLocator.getNonFatalsManager();
            if (nonFatalsManager != null) {
                nonFatalsManager.saveNonFatal(nonFatal);
            }
            return true;
        });
    }

    @VisibleForTesting
    @Nullable
    static StackTraceElement getFirstInstabugElementFromStacktrace(StackTraceElement[] stackTrace) {
        if (stackTrace != null && stackTrace.length > 0) {
            for (StackTraceElement element : stackTrace) {
                if (element.getClassName().startsWith(INSTABUG_PREFIX) && !(element.getMethodName().equals(METHOD_REPORT_ERROR) || element.getMethodName().equals(METHOD_REPORT_NON_FATAL))) {
                    return element;
                }
            }
        }
        return null;
    }

    private static boolean canReportNonFatal() {
        return !PerSessionSettings.getInstance().isNonFatalsDisabledTemporarily();
    }
}
