package com.instabug.library.diagnostics.customtraces.cache

import com.instabug.library.Constants
import com.instabug.library.diagnostics.customtraces.model.IBGCustomTrace
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.COLUMN_DURATION
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.COLUMN_END_BG
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.COLUMN_ID
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.COLUMN_NAME
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.COLUMN_START_BG
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.COLUMN_START_TIME
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.DURATION_DEFAULT_VALUE
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.TABLE_NAME
import com.instabug.library.diagnostics.diagnostics_db.CustomTracesEntry.TRIM_TO_LIMIT_WHERE_CLAUSE
import com.instabug.library.diagnostics.diagnostics_db.DiagnosticsDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.asBoolean
import com.instabug.library.util.extenstions.asInt

class CustomTracesDBHelperImpl : CustomTracesDBHelper {

    val database: DiagnosticsDbManager = DiagnosticsDbManager.getInstance()

    override fun insertTrace(
        trace: IBGCustomTrace
    ): Long {
        return database.let {
            val contentValues = IBGContentValues().apply {
                put(COLUMN_NAME, trace.name, ColumnsTransitiveState.NAME)
                put(COLUMN_START_TIME, trace.startTime, ColumnsTransitiveState.START_TIME)
                put(COLUMN_START_BG, trace.startedInBG.asInt, ColumnsTransitiveState.START_BG)
                put(COLUMN_END_BG, trace.endedInBG.asInt, ColumnsTransitiveState.END_BG)
                put(COLUMN_DURATION, trace.duration, ColumnsTransitiveState.DURATION)
            }

            it.insert(TABLE_NAME, null, contentValues)
        }.also {
            InstabugSDKLogger.v(
                Constants.LOG_TAG,
                "Started custom trace ${trace.name} with id: $it"
            )
        }
    }

    override fun endTrace(traceId: Long, duration: Long, endedInBG: Boolean): Boolean {
        return database.let {
            val contentValues = IBGContentValues().apply {
                put(COLUMN_DURATION, duration, ColumnsTransitiveState.DURATION)
                put(COLUMN_END_BG, endedInBG.asInt, ColumnsTransitiveState.END_BG)
            }
            val whereClause =
                "$COLUMN_ID = ? AND $COLUMN_DURATION = $DURATION_DEFAULT_VALUE"
            val whereArgs = listOf(IBGWhereArg(traceId.toString(), ColumnsTransitiveState.TRACE_ID))
            val affectedRowsCount = it.update(
                TABLE_NAME,
                contentValues,
                whereClause,
                whereArgs
            )
            affectedRowsCount > 0
        }.also {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Custom trace with id $traceId has ended")
        } ?: false
    }

    override fun removeAll() {
        database.delete(TABLE_NAME, null, null)
    }

    override fun removeUnEndedTraces() {
        database.let {
            val whereClause = "$COLUMN_DURATION = $DURATION_DEFAULT_VALUE"

            it.delete(
                TABLE_NAME,
                whereClause,
                null
            )
        }
    }

    override fun trimToLimit(limit: Int) {
        database.let {

            val whereArgs: MutableList<IBGWhereArg> = ArrayList()
            whereArgs.add(IBGWhereArg("-1", true))
            whereArgs.add(IBGWhereArg(limit.toString(), true))
            it.delete(
                TABLE_NAME,
                TRIM_TO_LIMIT_WHERE_CLAUSE,
                whereArgs
            )
        }
    }

    override fun getAllTraces(): List<IBGCustomTrace> {
        val traces = mutableListOf<IBGCustomTrace>()
        database.let {
            it.query(TABLE_NAME, null, null, null, null, null, null)
                ?.use { cursor: IBGCursor ->
                    while (cursor.moveToNext()) {
                        val customTrace = IBGCustomTrace(
                            id = cursor.getLong(cursor.getColumnIndex(COLUMN_ID)),
                            name = cursor.getString(cursor.getColumnIndex(COLUMN_NAME)),
                            startTime =
                            cursor.getLong(cursor.getColumnIndex(COLUMN_START_TIME)),
                            duration = cursor.getLong(cursor.getColumnIndex(COLUMN_DURATION)),
                            startedInBG =
                            cursor.getInt(cursor.getColumnIndex(COLUMN_START_BG)).asBoolean,
                            endedInBG =
                            cursor.getInt(cursor.getColumnIndex(COLUMN_END_BG)).asBoolean,
                        )
                        traces.add(customTrace)
                    }
                }
        }
        return traces
    }


    override fun deleteTraces(ids: List<Long>) {
        ids
            .takeUnless { it.isEmpty() }
            ?.also { database.beginTransaction() }
            ?.onEach(::deleteById)
            ?.also { database.endTransaction() }
    }

    private fun deleteById(id: Long) {
        database.delete(
            TABLE_NAME,
            "$COLUMN_ID = ?",
            listOf(IBGWhereArg(id.toString(), ColumnsTransitiveState.TRACE_ID))
        )
    }

    override fun getTraceId(trace: IBGCustomTrace): Long {
        var id = -1L
        database.let {
            val selection = "$COLUMN_NAME = ? AND $COLUMN_START_TIME = ? AND $COLUMN_DURATION = ?"
            val selectionsArgs = listOf<IBGWhereArg>(
                IBGWhereArg(trace.name, ColumnsTransitiveState.NAME),
                IBGWhereArg("${trace.startTime}", ColumnsTransitiveState.START_TIME),
                IBGWhereArg("${trace.duration}", ColumnsTransitiveState.DURATION),
            )
            it.query(
                TABLE_NAME,
                arrayOf(COLUMN_ID),
                selection,
                selectionsArgs,
                null,
                null,
                null
            )
                ?.use { cursor ->
                    if (cursor.moveToFirst()) {
                        id = cursor.getLong(cursor.getColumnIndex(COLUMN_ID))
                    }
                }
            return id
        }
    }

    override fun deleteTracesByName(tracesNames: List<String>) {
        tracesNames
            .filter { it.isNotBlank() }
            .takeUnless { it.isEmpty() }
            ?.also { database.beginTransaction() }
            ?.onEach(::deleteByName)
            ?.also { database.endTransaction() }
    }

    private fun deleteByName(name: String) {
        database.delete(
            TABLE_NAME,
            "$COLUMN_NAME = ?",
            listOf(IBGWhereArg(name, ColumnsTransitiveState.NAME))
        )
    }
}
