package com.instabug.library;

import android.annotation.SuppressLint;
import android.content.Context;

import com.instabug.library.model.LogDescriptor;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.DeviceStateProvider;

import java.lang.ref.WeakReference;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

/**
 * A class to provide session details, it will act as a middleware between other provider and the
 * layer that request these information.
 * <p>
 * *************
 * DEPENDENCIES
 * *************
 * 1. {@link SettingsManager}
 * 2. {@link DeviceStateProvider}
 * 3. {@link LogDescriptor} to provide session in this descriptor
 */
class SessionDetailsProvider {

    @Nullable
    private WeakReference<Context> contextWeakReference;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    SessionDetailsProvider(@Nullable Context context) {
        this.contextWeakReference = new WeakReference<>(context);
    }

    /**
     * A method to provide current app token
     *
     * @return app token
     */
    @VisibleForTesting
    @Nullable
    String provideAppToken() {
        return SettingsManager.getInstance().getAppToken();
    }

    /**
     * A method to provide sdk version
     *
     * @return current sdk version
     */
    @VisibleForTesting
    String provideSdkVersion() {
        return DeviceStateProvider.getSdkVersion();
    }

    /**
     * A method to provide os version
     *
     * @return current device os version.
     */
    @VisibleForTesting
    String provideOsVersion() {
        return DeviceStateProvider.getOS();
    }


    /**
     * A method to return free storage
     *
     * @return free storage or -1 if context not provided.
     */
    @VisibleForTesting
    long provideFreeStorage() {
        if (contextWeakReference != null) {
            Context context = contextWeakReference.get();
            if (context != null) {
                return DeviceStateProvider.getFreeMemory(context);
            }
        }
        return -1;
    }

    LogDescriptor getSessionDetails() {
        try {
            LogDescriptor.Builder logDescriptorBuilder = new LogDescriptor.Builder()
                    .setSdkVersion(provideSdkVersion())
                    .setOsVersion(provideOsVersion())
                    .setFreeMemory(provideFreeStorage());
            String appToken = provideAppToken();
            if (appToken != null) {
                logDescriptorBuilder.setAppToken(appToken);
            }
            return logDescriptorBuilder.build();
        } finally {
            contextWeakReference = null;
        }

    }


}
