package com.instabug.library.view;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.util.AttributeSet;
import com.instabug.library.R;
import com.instabug.library.R.styleable;
import com.instabug.library.settings.PerSessionSettings;

import androidx.annotation.ColorInt;
import androidx.annotation.Nullable;
import androidx.core.content.res.ResourcesCompat;

import static android.util.TypedValue.COMPLEX_UNIT_DIP;

public class IconView extends androidx.appcompat.widget.AppCompatTextView {

    public static final int ANDROID_ACTIONBAR_ICON_SIZE_DP = 24;
    public static final int ANDROID_ACTIONBAR_ICON_SIZE_PADDING_DP = 1;


    @ColorInt
    int backgroundColor = Color.TRANSPARENT;
    float strokeWidth = 0F;
    @ColorInt
    int strokeColor = Color.TRANSPARENT;
    @Nullable
    Paint paint;

    public IconView(Context context) {
        this(context, null);
    }

    public IconView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public IconView(Context context, @Nullable AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);

        // prevent exception in Android Studio / ADT interface builder
        if (this.isInEditMode()) {
            return;
        }

        Typeface font = ResourcesCompat.getFont(context, R.font.ibg_font_icons);

        setTypeface(font);

        final TypedArray array = context.obtainStyledAttributes(attrs, styleable.IconView);

        if (!array.hasValue(styleable.IconView_android_textSize)) {
            setActionBarSize();
        }

        if (!array.hasValue(styleable.IconView_android_padding)) {
            setActionBarPadding();
        }

        if (!array.hasValue(styleable.IconView_android_textColor)) {
            setTextColor(PerSessionSettings.getInstance().getPrimaryColor());
        }

        final int iconEnumValue = array.getInt(styleable.IconView_instabug_icon, -1);

        if (iconEnumValue != -1) {
            setText(IconViewUtils.getIconUnicode(iconEnumValue));
        }
        array.recycle();

        paint = new Paint(Paint.ANTI_ALIAS_FLAG);

    }

    private void setActionBarSize() {
        setTextSize(COMPLEX_UNIT_DIP, ANDROID_ACTIONBAR_ICON_SIZE_DP);
    }

    private void setActionBarPadding() {
        int padding = ViewUtils.convertDpToPx(getContext(), ANDROID_ACTIONBAR_ICON_SIZE_PADDING_DP);
        setPadding(padding);
    }

    public void setPadding(int padding) {
        setPadding(padding, padding, padding, padding);
    }


    @Override
    public void setBackgroundColor(@ColorInt int color) {
        this.backgroundColor = color;
        invalidate();
    }

    public void setStrokeWidth(float strokeWidth) {
        this.strokeWidth = strokeWidth;
        invalidate();
    }

    public void setStrokeColor(@ColorInt int strokeColor) {
        this.strokeColor = strokeColor;
        invalidate();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (paint != null) {
            paint.setColor(backgroundColor);
            paint.setStyle(Paint.Style.FILL);
            canvas.drawCircle(getWidth() / 2f, getHeight() / 2f, (Math.min(getWidth() /
                    2f, getHeight() / 2f) - (strokeWidth / 2f)), paint);
            paint.setStrokeWidth(strokeWidth);
            paint.setColor(strokeColor);
            paint.setStyle(Paint.Style.STROKE);
            canvas.drawCircle(getWidth() / 2f, getHeight() / 2f, (Math.min(getWidth() /
                    2f, getHeight()
                    / 2f) - (strokeWidth / 2f)), paint);
        }
        super.onDraw(canvas);
    }
}
