package com.instabug.library.util.overairversion

import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.overairversion.OverAirSharedConstants.CODE_PUSH_LENGTH_LIMIT
import com.instabug.library.util.overairversion.OverAirSharedConstants.CODE_PUSH_LIMIT_EXCEEDED_ERROR
import com.instabug.library.util.overairversion.OverAirSharedConstants.CODE_PUSH_NOT_PROVIDED_ERROR
import com.instabug.library.util.overairversion.OverAirSharedConstants.PREF_KEY_IB_OVER_AIR_VERSION
import com.instabug.library.util.overairversion.OverAirSharedConstants.PREF_VALUE_NOT_SET
import com.instabug.library.util.overairversion.OverAirSharedConstants.WITH_CODE_PUSH_FORMAT

object CodePushVersionHandler : OnAirVersionTypeHandler {

    @VisibleForTesting
    var storedCodePushVersion by corePref(PREF_KEY_IB_OVER_AIR_VERSION, PREF_VALUE_NOT_SET)

    /**
     * Stores the provided Code Push or clear the stored one if the new one is not valid.
     * validation criteria includes not being null or blank (consists only of whitespaces).
     * @param version the newly provided Code push version.
     * @param type which type to detect the corresponding handler if ExpoVersionHandler or CodePushHandler
     * @param isSetByUser Indicates whether the user have used the API to set the provided [version]
     * or didn't use it at all.
     */
    override fun handleVersion(version: String?, type: Int, isSetByUser: Boolean?) {
        version?.takeIf { OverAirSharedConstants.isValid(version) }
            ?.let(this::trimAndLogIfNeeded)
            ?.also(this::storeVersion)
            ?.also { OverAirSharedConstants.storedType = type }
            ?: clearVersionAndLog(isSetByUser ?: false)
    }

    /**
     * sanitize code push version format to be linked with device provider
     * @param version the provided code push version.
     */
    override fun sanitize(version: String): String =
        storedCodePushVersion.takeUnless { it == PREF_VALUE_NOT_SET }
            ?.let { storedVersion -> WITH_CODE_PUSH_FORMAT.format(version, storedVersion) }
            ?: version

    /**
     * Trims the provided Code Push version if its length exceeds the specified limit (currently 30 chars).
     * In addition, logs an error notifying the user that the provided version exceeds the specified
     * limit and a trimmed version shall be used instead.
     * @param version the provided Code Push version
     * @return the [String] provided in the input if its length didn't exceed the specified limit or
     * a [String] represents the first n chars of the input (trimmed version).
     */
    private fun trimAndLogIfNeeded(version: String): String =
        version.trim().let { trimmed ->
            if (trimmed.length <= CODE_PUSH_LENGTH_LIMIT) return trimmed
            InstabugSDKLogger.w(Constants.LOG_TAG, CODE_PUSH_LIMIT_EXCEEDED_ERROR)
            return trimmed.substring(0, CODE_PUSH_LENGTH_LIMIT)
        }

    /**
     * Clears the stored Code Push version and logs an error notifying the user that there's no
     * Code Push version provided and we'll proceed with the default app version if the user has initially
     * used the API to set a Code Push version.
     */
    private fun clearVersionAndLog(isSetByUser: Boolean) {
        if (isSetByUser) {
            InstabugSDKLogger.w(Constants.LOG_TAG, CODE_PUSH_NOT_PROVIDED_ERROR)
        }
        storedCodePushVersion = PREF_VALUE_NOT_SET
    }

    /**
     * Stores the provided Code Push version to be delegate to sharedpref.
     * @param version the provided Code Push version.
     */
    private fun storeVersion(version: String) {
        storedCodePushVersion = version
    }
}
