package com.instabug.library.util;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.internal.storage.AttachmentManager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import io.reactivexport.Observable;
import io.reactivexport.schedulers.Schedulers;

/**
 * @author hossam.
 */

public class DiskUtils {

    @Nullable
    public static synchronized Uri zipFiles(Context context, String fileName,
                                            ArrayList<File> files) {

        InstabugSDKLogger.v(Constants.LOG_TAG, "zipping " + files.size() + " files ... ");

        OutputStream outputStream = null;
        ZipOutputStream zipOutputStream = null;
        try {
            //create new file .zip
            File zipFile = new File(AttachmentManager.getAttachmentInternalDirectory((context)) + File
                    .separator + fileName + "_" + System.currentTimeMillis() + ".zip");

            outputStream = new FileOutputStream(zipFile);
            zipOutputStream = new ZipOutputStream(outputStream);

            for (File file : files) {
                if (file != null) {
                    zipOutputStream.putNextEntry(new ZipEntry(file.getName()));
                    copy(file, zipOutputStream);
                    zipOutputStream.closeEntry();
                }
            }

            InstabugSDKLogger.v(Constants.LOG_TAG, "zipping files have been finished successfully, " +
                    "path: " + zipFile.getPath());
            return Uri.fromFile(zipFile);
        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "zipping files went wrong: " + e
                    .getMessage() + ", time in MS: " + System.currentTimeMillis(), e);
            return null;
        } finally {
            if (zipOutputStream != null) {
                try {
                    zipOutputStream.close();
                } catch (IOException e) {
                    // Swallow
                }
            }
            if (outputStream != null) {
                try {
                    outputStream.close();
                } catch (IOException e) {
                    // Swallow
                }
            }
        }
    }

    private static void copy(File file, OutputStream out) throws IOException {
        InputStream in = null;
        try {
            in = new FileInputStream(file);
            copy(in, out);
        } catch (FileNotFoundException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG,
                    "FileNotFoundException: can not copy file to another stream");
        } finally {
            if (in != null) {
                in.close();
            }
        }
    }

    private static void copy(InputStream in, OutputStream out) throws IOException {
        byte[] buffer = new byte[2048];
        while (true) {
            int readCount = in.read(buffer);
            if (readCount < 0) {
                break;
            }
            out.write(buffer, 0, readCount);
        }
    }

    public static ArrayList<File> listFilesInDirectory(File directory) {
        File[] files = directory.listFiles();
        List<File> fileList = new ArrayList<>();
        if (files != null) {
            fileList = Arrays.asList(files);
        }
        return new ArrayList<>(fileList);
    }


    @NonNull
    public static Observable<List<File>> getCleanDirectoryObservable(final File directory) {
        return Observable.fromCallable(() -> {
            List<File> unDeletedFiles = new ArrayList<>();
            if (directory.exists() && directory.isDirectory()) {
                File[] files = directory.listFiles();
                if (files != null) {
                    for (File file : files) {
                        if (!file.delete()) {
                            unDeletedFiles.add(file);
                        }
                    }
                }
            }
            return unDeletedFiles;
        }).subscribeOn(Schedulers.io());
    }

    @NonNull
    public static Observable<List<File>> getDeleteDirectoryObservable(final File directory) {
        return Observable.fromCallable(() -> {
            List<File> unDeletedFiles = new ArrayList<>();
            if (directory.exists() && directory.isDirectory()) {
                File[] files = directory.listFiles();
                if (files != null) {
                    for (File file : files) {
                        if (!file.delete()) {
                            unDeletedFiles.add(file);
                        }
                    }
                }
            }
            if (unDeletedFiles.isEmpty()) {
                directory.delete();
            }
            return unDeletedFiles;
        }).subscribeOn(Schedulers.io());
    }
}
