package com.instabug.library.tracking

import android.app.Activity
import java.util.concurrent.ConcurrentHashMap

interface ActivityMetaDataProvider {
    fun addActivityMetaData(activity: Activity)
    fun getActivitySimpleName(activity: Activity): String
    fun getActivityName(activity: Activity): String
    fun clearActivityMetaData(activity: Activity)
    fun isLastRemainingActivity(activity: Activity): Boolean
}

class ActivityMetaDataProviderImpl(
    private val activityMetaDataMap: MutableMap<Int, ActivityMetaData> = ConcurrentHashMap()
) : ActivityMetaDataProvider {

    override fun addActivityMetaData(activity: Activity) {
        val activityId = activity.systemHashCode()
        if (!activityMetaDataMap.containsKey(activityId)) {
            activityMetaDataMap[activityId] = ActivityMetaData.from(activity)
        }
    }

    override fun getActivitySimpleName(activity: Activity): String {
        return activityMetaDataMap[activity.systemHashCode()]?.simpleName.orEmpty()
    }

    override fun getActivityName(activity: Activity): String {
        return activityMetaDataMap[activity.systemHashCode()]?.activityName.orEmpty()
    }

    override fun clearActivityMetaData(activity: Activity) {
        activityMetaDataMap.remove(activity.systemHashCode())
    }

    override fun isLastRemainingActivity(activity: Activity): Boolean {
        return activityMetaDataMap.size == 1 && activityMetaDataMap.containsKey(activity.systemHashCode())
    }

    private fun Activity.systemHashCode(): Int {
        return System.identityHashCode(this)
    }

    class ActivityMetaData(val simpleName: String, val activityName: String) {
        companion object {
            fun from(activity: Activity): ActivityMetaData {
                return ActivityMetaData(
                    activity.javaClass.simpleName.orEmpty(),
                    activity.javaClass.name.orEmpty()
                )
            }
        }
    }
}
