package com.instabug.library.sessionreplay

import android.net.Uri
import com.instabug.library.core.InstabugCore
import com.instabug.library.internal.filestore.FileOperation
import com.instabug.library.internal.filestore.NEWLINE_BYTE_VALUE
import com.instabug.library.sessionreplay.bitmap.BitmapCompressor
import com.instabug.library.sessionreplay.model.SRLog
import com.instabug.library.sessionreplay.model.SRScreenshotLog
import com.instabug.library.sessionreplay.monitoring.ErrorCompressingSRLogsFile
import com.instabug.library.sessionreplay.monitoring.ErrorCreatingSRLogsFile
import com.instabug.library.sessionreplay.monitoring.ErrorCreatingSRSessionDirectory
import com.instabug.library.sessionreplay.monitoring.ErrorStoringSRScreenshot
import com.instabug.library.sessionreplay.monitoring.ErrorWritingInSRLogsFile
import com.instabug.library.sessionreplay.monitoring.ErrorZippingSRScreenshots
import com.instabug.library.util.extenstions.createNewFileDefensive
import com.instabug.library.util.extenstions.deleteDefensive
import com.instabug.library.util.extenstions.ifNotExists
import com.instabug.library.util.extenstions.mkdirsDefensive
import com.instabug.library.util.extenstions.takeIfExists
import com.instabug.library.util.extenstions.write
import java.io.BufferedOutputStream
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.util.zip.GZIPOutputStream
import java.util.zip.ZipEntry
import java.util.zip.ZipOutputStream

class WriteSRLogOperation(private val log: SRLog) :
    FileOperation<SRSessionDirectory, Int> {
    @Throws(ErrorWritingInSRLogsFile::class)
    override fun invoke(input: SRSessionDirectory): Int {
        val logBytes = log.srJsonRep?.toString()?.toByteArray() ?: return 0
        return runCatching {
            input.logsFile.apply { parentFile?.ifNotExists { mkdirsDefensive() } }
                .ifNotExists { createNewFileDefensive() }
                .takeIfExists()
                ?.let { FileOutputStream(it, true) }
                ?.use { output -> with(output) { write(logBytes); write(NEWLINE_BYTE_VALUE) } }
                ?.let { logBytes.size + 1 }
                ?: throw ErrorCreatingSRLogsFile(input.logsFile.toString())
        }.onFailure { t -> throw ErrorWritingInSRLogsFile(t) }.getOrThrow()
    }
}

val CompressLogsFileOperation = FileOperation<SRSessionDirectory, Unit> { directory ->
    runCatching {
        directory.logsFile.takeIf(File::exists)?.inputStream()?.use { inputStream ->
            directory.compressedLogsFile.apply { parentFile?.ifNotExists { mkdirsDefensive() } }
                .ifNotExists { createNewFileDefensive() }
                .takeIfExists()
                ?.let(::FileOutputStream)
                ?.let(::GZIPOutputStream)
                ?.use { outputStream -> outputStream.write(inputStream) }
                ?: throw ErrorCreatingSRLogsFile(directory.compressedLogsFile.toString())
        }
        directory.logsFile.takeIf(File::exists)?.deleteDefensive()
    }.onFailure { t -> throw ErrorCompressingSRLogsFile(t) }.getOrThrow()
}

class ZipSRScreenshots : FileOperation<SRSessionDirectory, Unit> {
    @Throws(ErrorZippingSRScreenshots::class)
    override fun invoke(input: SRSessionDirectory): Unit = runCatching {
        with(input) {
            screenshots.listFiles()
                ?.filterNot { it.isDirectory }
                ?.takeUnless { it.isEmpty() }
                ?.let { screenShotsFiles -> screenshotsZipFile.addFiles(screenShotsFiles) }
                ?.also { screenshots.deleteRecursively() }; Unit
        }
    }.onFailure { t -> throw ErrorZippingSRScreenshots(t) }.getOrThrow()

    @Throws(ErrorCreatingSRSessionDirectory::class)
    private fun File.zipOutputStream() =
        apply { parentFile?.ifNotExists { mkdirsDefensive() } }
            .ifNotExists { createNewFileDefensive() }
            .takeIfExists()
            ?.let(::FileOutputStream)
            ?.let(::ZipOutputStream)
            ?: throw ErrorCreatingSRLogsFile(this.toString())

    private fun File.addFiles(
        files: List<File>
    ) = zipOutputStream().use { zipOutStream ->
        files.forEach { file -> zipOutStream.addFile(file) }
    }

    private fun ZipOutputStream.addFile(file: File) {
        val input = FileInputStream(file)
        putNextEntry(ZipEntry(file.name))
        input.use { write(input) }
    }
}

class SaveScreenshotOp(
    private val screenshot: SRScreenshotLog,
    private val compressor: BitmapCompressor
) : FileOperation<SRSessionDirectory, Long> {
    @Throws(ErrorStoringSRScreenshot::class)
    override fun invoke(input: SRSessionDirectory): Long =
        File(input.screenshots, screenshot.fileName).runCatching {
            parentFile?.ifNotExists { mkdirsDefensive() }
                ?.ifNotExists { throw ErrorCreatingSRSessionDirectory(this.toString()) }
            this.let(::FileOutputStream)
                .let(::BufferedOutputStream)
                .use { outStream -> saveScreenshot(outStream, this) }
            screenshot.nullifyBitmap()
            ifNotExists { throw ErrorCreatingSRLogsFile(this.toString()) }
            Uri.fromFile(this).path?.also(InstabugCore::encryptBeforeMarshmallow)
            length()
        }.onFailure { t -> throw ErrorStoringSRScreenshot(t) }.getOrThrow()

    @Throws
    private fun saveScreenshot(outStream: BufferedOutputStream, savingFile: File): Unit? =
        screenshot.bitmap
            ?.runCatching { compressor(this, outStream) }
            ?.onFailure { savingFile.deleteDefensive() }
            ?.getOrThrow()
}

class DeleteSessionDirectoryOperation(private val sessionId: String) :
    FileOperation<File, Unit> {
    override fun invoke(input: File) {
        SRSessionDirectory(sessionId, input).deleteFilesDirectory()
    }
}

class GetSessionDirectoryOperation(
    private val sessionId: String
) : FileOperation<File, SRSessionDirectory?> {
    override fun invoke(input: File): SRSessionDirectory? =
        SRSessionDirectory(sessionId, input)
            .takeIf { dir -> dir.filesDirectory.exists() }
}

class CompressSRDirsOperation(
    private val compressor: FileOperation<SRSessionDirectory, Unit> = CompressLogsFileOperation,
    private val screenshotZipper: FileOperation<SRSessionDirectory, Unit> = ZipSRScreenshots()
) : FileOperation<List<SRSessionDirectory>, List<String>> {
    override fun invoke(input: List<SRSessionDirectory>) = input
        .onEach { sessionDir -> compressor(sessionDir) }
        .onEach { sessionDir -> screenshotZipper(sessionDir) }
        .map { it.sessionId }
}

class MapSessionsAndExecuteOperation(
    private val sessionsIds: List<String>,
    private val operation: FileOperation<List<SRSessionDirectory>, List<String>>
) : FileOperation<File, List<String>> {
    override fun invoke(input: File) = sessionsIds
        .map { spanDir -> SRSessionDirectory(spanDir, input) }
        .let(operation::invoke)
}

class CalculateSRAggregateSizeOperation : FileOperation<List<SRSessionDirectory>, Long> {
    override fun invoke(input: List<SRSessionDirectory>): Long =
        input.fold(0L) { acc, directory -> acc + getCountableSizeInADirectory(directory) }

    private fun getCountableSizeInADirectory(directory: SRSessionDirectory): Long =
        getLogsFileSize(directory) + getScreenshotsSize(directory)

    private fun getLogsFileSize(directory: SRSessionDirectory): Long =
        (getUncompressedLogsFileSize(directory) ?: getCompressedLogsFileSize(directory)) ?: 0L

    private fun getUncompressedLogsFileSize(directory: SRSessionDirectory): Long? =
        directory.logsFile.takeIf(File::exists)?.length()

    private fun getCompressedLogsFileSize(directory: SRSessionDirectory): Long? =
        directory.compressedLogsFile.takeIf(File::exists)?.length()

    private fun getScreenshotsSize(directory: SRSessionDirectory): Long =
        (getScreenshotsDirSize(directory) ?: getScreenshotsZipSize(directory)) ?: 0L

    private fun getScreenshotsDirSize(directory: SRSessionDirectory): Long? =
        directory.screenshots.takeIf(File::exists)?.listFiles()
            ?.fold(0L) { acc, file -> acc + file.length() }

    private fun getScreenshotsZipSize(directory: SRSessionDirectory): Long? =
        directory.screenshotsZipFile.takeIf(File::exists)?.length()
}

