package com.instabug.library.sessioncontroller

import androidx.annotation.MainThread

/**
 * Listener for Instabug session lifecycle events.
 *
 * Use this to get notified when a session starts or stops via
 * [com.instabug.library.Instabug.startSession] and [com.instabug.library.Instabug.stopSession].
 *
 * All methods run on the main thread. Avoid heavy work to prevent UI blocking.
 *
 * ### Example:
 * ```kotlin
 * Instabug.setSessionLifeCycleCallBack(object : IBGSessionLifecycleCallback {
 *     override fun onSessionStart() { /* UI-safe code */ }
 *     override fun onSessionStop() { /* UI-safe code */ }
 * })
 * ```
 *
 * @see com.instabug.library.Instabug.setSessionLifeCycleCallBack
 */

interface IBGSessionLifecycleCallback {

    /**
     * Called when an Instabug session starts via [com.instabug.library.Instabug.startSession].
     *
     * Runs on the main thread — safe for UI updates.
     * Avoid heavy work to prevent blocking the UI; use background threads if needed.
     */

    @MainThread
    fun onSessionStart()

    /**
     * Called when an Instabug session ends via [com.instabug.library.Instabug.stopSession].
     *
     * Runs on the main thread — safe for UI updates.
     * Avoid heavy work to prevent blocking the UI; use background threads if needed.
     */

    @MainThread
    fun onSessionStop()
}
