package com.instabug.library.sessionV3.sync

import com.instabug.library.Constants.LOG_TAG
import com.instabug.library.core.plugin.PluginsManager
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.featuresflags.constants.FFMode
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator
import com.instabug.library.featuresflags.mappers.featuresFlagsAsExperiments
import com.instabug.library.featuresflags.mappers.featuresFlagsAsJson
import com.instabug.library.featuresflags.mappers.toExperimentsJsonString
import com.instabug.library.model.v3Session.IBGSession
import com.instabug.library.model.v3Session.IBGSessionDTO
import com.instabug.library.model.v3Session.IBGSessionData
import com.instabug.library.model.v3Session.IBGSessionMap
import com.instabug.library.model.v3Session.IBGSessionMapper.asPair
import com.instabug.library.model.v3Session.IBGSessions
import com.instabug.library.model.v3Session.SessionsExperimentsKeys.SESSION_EXPERIMENTS_KEY
import com.instabug.library.model.v3Session.SessionsFeaturesFlagsKeys.SESSION_FEATURES_FLAGS_KEY
import com.instabug.library.model.v3Session.SyncStatus
import com.instabug.library.sessionV3.cache.id
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.providers.SessionID
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.util.extenstions.toJsonArray
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONArray


interface SessionDataController {
    fun collectSessions(): IBGSessionDTO?
    fun deleteSessions(sessionsIds: List<String>)
    fun deleteSyncedSessions()
}

object SessionDataControllerImpl : SessionDataController {
    private val sessionsCacheManager by lazy { IBGSessionServiceLocator.sessionCacheManger }
    private val configurations by lazy { IBGSessionServiceLocator.sessionConfigurations }
    private val sessionsMerger by lazy { IBGSessionServiceLocator.sessionMerger }
    private val featuresFlagsCacheManager by lazy { IBGSessionServiceLocator.featuresFlagsCacheManager }
    private val featureSessionDataControllers
        get() = PluginsManager.getFeaturesSessionDataControllers()
    private val featureSessionLazyDataProviders
        get() = PluginsManager.getFeaturesSessionLazyDataProvider()

    override fun collectSessions(): IBGSessionDTO? {
        val sessions = getSessions()
        return sessions
            .also { logReadyForSyncSessions(it) }
            ?.toMap()
            ?.let { sessionsMap -> sessionsMerger.mergeCommon(sessionsMap) }
            ?.also { dto -> dto.sessionReplayCount = sessions?.count { it.srEnabled } ?: 0 }
    }

    private fun logReadyForSyncSessions(ibgSessions: List<IBGSession>?) {
        if (ibgSessions.isNullOrEmpty())
            InstabugSDKLogger.d(LOG_TAG, "No sessions ready for sync. Skipping...")
        else
            InstabugSDKLogger.d(LOG_TAG, "Synced a batch of ${ibgSessions.size} session/s.")

    }

    private fun IBGSessions.toMap(): List<IBGSessionMap> {
        val sessionsSerials = map { session -> session.serial }
        val featuresSessionsData = extractFeaturesSessionsData()
        extractFeaturesKeys(featuresSessionsData).let(sessionsMerger::addUniqueKeys)
        val featuresFlags = featuresFlagsCacheManager.getFeaturesFlags(sessionsSerials)
        return map { session ->
            session.toMap(
                featuresData = featuresSessionsData,
                sessionFeaturesFlags = featuresFlags[session.serial]
            )
        }
    }

    private fun IBGSessions.extractFeaturesSessionsData(): List<Map<SessionID, IBGSessionData>> {
        val sessionsIds = map { session -> session.id }
        return runOrLogError(errorMessage = "couldn't collect data from other modules ") {
            featureSessionDataControllers
                .map { PoolProvider.submitIOTask { it.collectSessionsData(sessionsIds) } }
                .map { future -> future.get() }
        }.onFailure {
            IBGDiagnostics.reportNonFatal(
                it,
                "error while collecting data from other modules"
            )
        }.getOrElse { emptyList() }
    }


    private fun extractFeaturesKeys(featuresData: List<Map<SessionID, IBGSessionData>>): Set<String> =
        featuresData.asSequence()
            .flatMap { it.values }
            .map { it.featureKey }
            .toSet()


    private fun IBGSession.toMap(
        featuresData: List<Map<String, IBGSessionData>>,
        sessionFeaturesFlags: JSONArray?
    ): MutableMap<String, Any> {
        val sessionFeaturesData = extractSessionFeatureData(featuresData)
        return extractFields(hashMapOf()).apply {
            when (FeaturesFlagServiceLocator.featuresFlagsConfigsProvider.mode) {
                FFMode.EXPERIMENTS ->
                    sessionFeaturesFlags?.let {
                        this[SESSION_EXPERIMENTS_KEY] =
                            featuresFlagsAsExperiments(it).toJsonArray.toExperimentsJsonString()
                    }

                FFMode.FF ->
                    sessionFeaturesFlags
                        ?.takeIf { sessionFeaturesFlags.length() > 0 }
                        ?.let {
                            this[SESSION_FEATURES_FLAGS_KEY] = featuresFlagsAsJson(it).toString()
                        }
            }
            putAll(sessionFeaturesData)
        }
    }

    private fun IBGSession.extractSessionFeatureData(featuresData: List<Map<String, IBGSessionData>>) =
        featuresData
            .mapNotNull { featureSessionsMap -> featureSessionsMap[id] }
            .map { it.asPair }


    override fun deleteSessions(sessionsIds: List<String>) {
        sessionsCacheManager.changeSyncStatus(
            from = SyncStatus.READY_FOR_SYNC,
            to = SyncStatus.SYNCED,
            sessionsIds
        )
        deleteSessionsWithSessionsData(sessionsIds)
    }

    private fun deleteSessionsWithSessionsData(sessionsIds: List<String>) {
        featureSessionDataControllers
            .map { PoolProvider.submitIOTask { it.dropSessionData(sessionsIds) } }
            .forEach { future -> future.get() }
        sessionsCacheManager.deleteSessionByID(sessionsIds)
    }

    override fun deleteSyncedSessions() {
        sessionsCacheManager
            .querySessionsIdsBySyncStatus(SyncStatus.SYNCED)
            .map { sessionStatus -> sessionStatus.id }
            .let(::deleteSessionsWithSessionsData)
    }

    private fun getSessions(): IBGSessions? = sessionsCacheManager
        .querySessions(
            status = SyncStatus.READY_FOR_SYNC,
            limit = configurations.sessionRequestLimit
        )
        .takeUnless { it.isEmpty() }
        ?.takeUnless { it.isSessionReplayReady() }

    private fun IBGSessions.isSessionReplayReady() = any { session ->
        session.srEnabled && !session.isSrEvaluated
    }
}