package com.instabug.library.sessionV3.configurations

import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.InstabugFeaturesManager
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.model.v3Session.Defaults.DEFAULT_ANR_STORE_LIMIT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_DEBUG_MODE_ENABLED
import com.instabug.library.model.v3Session.Defaults.DEFAULT_EXPERIMENTS_ENABLED
import com.instabug.library.model.v3Session.Defaults.DEFAULT_EXPERIMENTS_STORE_LIMIT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_FATAL_HANG_STORE_LIMIT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_LAST_SYNC
import com.instabug.library.model.v3Session.Defaults.DEFAULT_NON_FATAL_STORE_LIMIT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_PERIODIC_DURATION_CAPTURE_ENABLED
import com.instabug.library.model.v3Session.Defaults.DEFAULT_PERIODIC_DURATION_CAPTURE_INTERVAL
import com.instabug.library.model.v3Session.Defaults.DEFAULT_SESSIONS_DROPPED_COUNT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_SESSIONS_REQUEST_LIMIT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_SESSIONS_STORE_LIMIT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_STITCHING_ENABLED
import com.instabug.library.model.v3Session.Defaults.DEFAULT_STITCHING_SESSION_TIMEOUT
import com.instabug.library.model.v3Session.Defaults.DEFAULT_SYNC_INTERVAL
import com.instabug.library.model.v3Session.Defaults.DEFAULT_V3_ENABLED
import com.instabug.library.model.v3Session.Defaults.DEFAULT_V3_PERCENTAGE
import com.instabug.library.model.v3Session.PreferencesKeys.ANR_STORE_LIMIT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.DEBUG_MODE_ENABLED_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.DROPPED_SESSIONS_COUNT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.EXPERIMENTS_ENABLED_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.EXPERIMENTS_STORE_LIMIT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.FATAL_HANG_STORE_LIMIT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.NON_FATAL_STORE_LIMIT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.PERIODIC_DURATION_CAPTURE_ENABLED
import com.instabug.library.model.v3Session.PreferencesKeys.PERIODIC_DURATION_CAPTURE_INTERVAL
import com.instabug.library.model.v3Session.PreferencesKeys.SESSIONS_LAST_SYNC_TIME_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.SESSIONS_REQUEST_LIMIT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.SESSIONS_STORE_LIMIT_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.SESSIONS_SYNC_INTERVAL_KEY
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_ANR_STORE_LIMIT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_DEBUG_MODE_ENABLED
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_EXPERIMENTS
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_EXPERIMENTS_ENABLED
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_EXPERIMENTS_STORE_LIMIT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_FATAL_HANG_STORE_LIMIT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_NON_FATAL_STORE_LIMIT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_PERIODIC_DURATION_CAPTURE_ENABLED
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_PERIODIC_DURATION_CAPTURE_INTERVAL
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_SESSIONS_REQUEST_LIMIT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_SESSIONS_STORE_LIMIT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_STITCHING_ENABLED
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_STITCHING_SESSION_TIMEOUT
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_SYNC_INTERVAL
import com.instabug.library.model.v3Session.SessionsFeaturesFlags.FLAG_V3_ENABLED
import com.instabug.library.percentagefeatures.IBGPercentageFlagsResolver
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject

interface IBGConfigurationsHandler {
    fun handle(sessionsConfig: JSONObject?)
    fun handlePeriodicUpdateFlags(responseJsonObject: JSONObject?)
}

interface IBGSessionConfigurations : IBGSessionCrashesConfigurations {
    val isV3SessionEnabled: Boolean
    var isDebugModeEnabled: Boolean
    val inDebugMode: Boolean
    var sessionsStoreLimit: Int
    var droppedSessionCount: Int
    var syncInterval: Long
    var sessionRequestLimit: Int
    var lastSyncTime: Long
    var isExperimentsEnabled: Boolean
    var experimentsStoreLimit: Int
    var periodicDurationCaptureEnabled: Boolean
    var periodicDurationCaptureInterval: Long
    fun incrementDroppedSessionCount(dropped: Int) {
        droppedSessionCount += dropped
    }
}

object IBGSessionConfigurationsImpl : IBGSessionConfigurations {
    override val isV3SessionEnabled: Boolean
        get() = SettingsManager.getInstance()
            .getFeatureState(IBGFeature.V3_SESSION, DEFAULT_V3_ENABLED) == Feature.State.ENABLED
                && InstabugFeaturesManager.getInstance()
                    .getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED
    override var isDebugModeEnabled: Boolean
            by corePref(DEBUG_MODE_ENABLED_KEY, DEFAULT_DEBUG_MODE_ENABLED)
    override val inDebugMode: Boolean
        get() = isDebugModeEnabled && IBGSessionServiceLocator.sessionDataProvider.isInDebugMode
    override var sessionsStoreLimit: Int
            by corePref(SESSIONS_STORE_LIMIT_KEY to DEFAULT_SESSIONS_STORE_LIMIT)
    override var droppedSessionCount: Int
            by corePref(DROPPED_SESSIONS_COUNT_KEY to DEFAULT_SESSIONS_DROPPED_COUNT)
    override var syncInterval: Long
            by corePref(SESSIONS_SYNC_INTERVAL_KEY, DEFAULT_SYNC_INTERVAL)
    override var sessionRequestLimit: Int
            by corePref(SESSIONS_REQUEST_LIMIT_KEY, DEFAULT_SESSIONS_REQUEST_LIMIT)
    override var lastSyncTime: Long
            by corePref(SESSIONS_LAST_SYNC_TIME_KEY, DEFAULT_LAST_SYNC)
    override var isExperimentsEnabled: Boolean
            by corePref(EXPERIMENTS_ENABLED_KEY, DEFAULT_EXPERIMENTS_ENABLED)
    override var experimentsStoreLimit: Int
            by corePref(EXPERIMENTS_STORE_LIMIT_KEY, DEFAULT_EXPERIMENTS_STORE_LIMIT)
    override var periodicDurationCaptureEnabled: Boolean
            by corePref(
                PERIODIC_DURATION_CAPTURE_ENABLED,
                DEFAULT_PERIODIC_DURATION_CAPTURE_ENABLED
            )
    override var periodicDurationCaptureInterval: Long
            by corePref(
                PERIODIC_DURATION_CAPTURE_INTERVAL,
                DEFAULT_PERIODIC_DURATION_CAPTURE_INTERVAL
            )
    override var nonFatalStoreLimit: Int
            by corePref(NON_FATAL_STORE_LIMIT_KEY, DEFAULT_NON_FATAL_STORE_LIMIT)
    override var anrStoreLimit: Int
            by corePref(ANR_STORE_LIMIT_KEY, DEFAULT_ANR_STORE_LIMIT)
    override var fatalHangStoreLimit: Int
            by corePref(FATAL_HANG_STORE_LIMIT_KEY, DEFAULT_FATAL_HANG_STORE_LIMIT)
}

object IBGSessionConfigurationsHandlerImpl : IBGConfigurationsHandler {
    private val configurations by lazy { IBGSessionServiceLocator.sessionConfigurations }
    private val stitchingManger by lazy { IBGSessionServiceLocator.stitchingManger }

    override fun handle(sessionsConfig: JSONObject?) {
        runOrLogError(errorMessage = "Can't parse V3 Session configurations") {
            sessionsConfig
                ?.handleConfigurations()
                ?.handleStitchingConfigurations()
                ?.handleExperiments()
        }
    }

    override fun handlePeriodicUpdateFlags(responseJsonObject: JSONObject?) {
        responseJsonObject?.runOrLogError(errorMessage = "Can't parse V3 Session experiments configurations") {
            val periodicCaptureEnabled = optBoolean(
                FLAG_PERIODIC_DURATION_CAPTURE_ENABLED,
                DEFAULT_PERIODIC_DURATION_CAPTURE_ENABLED
            )
            val periodicCaptureDuration = optLong(
                FLAG_PERIODIC_DURATION_CAPTURE_INTERVAL,
                DEFAULT_PERIODIC_DURATION_CAPTURE_INTERVAL
            )
            configurations.periodicDurationCaptureEnabled =
                periodicCaptureEnabled && periodicCaptureDuration > 0
            configurations.periodicDurationCaptureInterval = periodicCaptureDuration
        }
    }

    private fun JSONObject.handleConfigurations(): JSONObject = apply {
        with(configurations) {
            handlePercentageState(optDouble(FLAG_V3_ENABLED, DEFAULT_V3_PERCENTAGE))
            isDebugModeEnabled = optBoolean(FLAG_DEBUG_MODE_ENABLED, DEFAULT_DEBUG_MODE_ENABLED)
            syncInterval = optLong(FLAG_SYNC_INTERVAL, DEFAULT_SYNC_INTERVAL)
            sessionRequestLimit =
                optInt(FLAG_SESSIONS_REQUEST_LIMIT, DEFAULT_SESSIONS_REQUEST_LIMIT)
            sessionsStoreLimit = optInt(FLAG_SESSIONS_STORE_LIMIT, DEFAULT_SESSIONS_STORE_LIMIT)

            nonFatalStoreLimit =
                optPositiveInt(FLAG_NON_FATAL_STORE_LIMIT, DEFAULT_NON_FATAL_STORE_LIMIT)
            anrStoreLimit = optPositiveInt(FLAG_ANR_STORE_LIMIT, DEFAULT_ANR_STORE_LIMIT)
            fatalHangStoreLimit =
                optPositiveInt(FLAG_FATAL_HANG_STORE_LIMIT, DEFAULT_FATAL_HANG_STORE_LIMIT)
        }
    }

    private fun JSONObject.optPositiveInt(key: String, defaultValue: Int) =
        optInt(key, defaultValue)
            .takeUnless { value -> value < 0 }
            ?: defaultValue

    private fun JSONObject.handleStitchingConfigurations(): JSONObject = apply {
        with(stitchingManger) {
            isStitchingEnabled = optBoolean(FLAG_STITCHING_ENABLED, DEFAULT_STITCHING_ENABLED)
            sessionTimeoutInSeconds =
                optInt(FLAG_STITCHING_SESSION_TIMEOUT, DEFAULT_STITCHING_SESSION_TIMEOUT)
        }
    }

    private fun JSONObject.handleExperiments(): JSONObject = apply {
        optJSONObject(FLAG_EXPERIMENTS)
            ?.let { experimentsObj ->
                with(configurations) {
                    isExperimentsEnabled = experimentsObj.optBoolean(
                        FLAG_EXPERIMENTS_ENABLED,
                        DEFAULT_EXPERIMENTS_ENABLED
                    )
                    experimentsStoreLimit = experimentsObj.optInt(
                        FLAG_EXPERIMENTS_STORE_LIMIT,
                        DEFAULT_EXPERIMENTS_STORE_LIMIT
                    )
                }
            }

    }

    private fun handlePercentageState(enabled: Double) {
        IBGPercentageFlagsResolver.resolve(IBGFeature.V3_SESSION, enabled)
    }
}

interface IBGSessionCrashesConfigurations {
    var nonFatalStoreLimit: Int
    var anrStoreLimit: Int
    var fatalHangStoreLimit: Int
}