package com.instabug.library.performanceclassification

import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants
import com.instabug.library.IBGFeature
import com.instabug.library.Instabug
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.percentagefeatures.IBGPercentageFlagsResolver
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.AVERAGE
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.AVERAGE_DEVICES_CLASS_LIST_KEY
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.AVERAGE_TRIMMING_PERCENTAGE
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.DEFAULT_TRIMMING_PERCENTAGE
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.DEVICE_PERFORMANCE_CLASS_VALUE
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.HIGH
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.HIGH_DEVICES_CLASS_LIST_KEY
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.LOW
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.LOW_DEVICES_CLASS_LIST_KEY
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.LOW_TRIMMING_PERCENTAGE
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.PERFORMANCE_CLASS_AVG_LIMIT_FLAG
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.PERFORMANCE_CLASS_LOW_LIMIT_FLAG
import com.instabug.library.performanceclassification.DevicePerformanceClassConfig.Companion.UNDEFINED
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils.Companion.PERFORMANCE_CLASS_AVERAGE
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils.Companion.PERFORMANCE_CLASS_HIGH
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils.Companion.PERFORMANCE_CLASS_LOW
import com.instabug.library.performanceclassification.DevicePerformanceClassUtils.Companion.PERFORMANCE_CLASS_UNDEFINED
import com.instabug.library.settings.PerSessionSettings
import com.instabug.library.settings.PersistableSettings
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import org.json.JSONArray
import org.json.JSONObject

interface DevicePerformanceClassConfig {
    companion object {
        const val LOW_DEVICES_CLASS_LIST_KEY = "ibg_low_devices_performance_class"
        const val AVERAGE_DEVICES_CLASS_LIST_KEY = "ibg_average_devices_performance_class"
        const val HIGH_DEVICES_CLASS_LIST_KEY = "ibg_high_devices_performance_class"
        const val DEVICE_PERFORMANCE_CLASS_VALUE = "ibg_device_performance_class_value"
        const val LOW_TRIMMING_PERCENTAGE = "ibg_low_trimming_percentage"
        const val AVERAGE_TRIMMING_PERCENTAGE = "ibg_average_trimming_percentage"
        const val DEFAULT_TRIMMING_PERCENTAGE = 1.0F

        const val PERFORMANCE_CLASS_LOW_LIMIT_FLAG = "dv_perf_class_low_limit"
        const val PERFORMANCE_CLASS_AVG_LIMIT_FLAG = "dv_perf_class_avg_limit"

        //Performance class literals
        const val LOW = "LOW"
        const val AVERAGE = "AVERAGE"
        const val HIGH = "HIGH"
        const val UNDEFINED = "UNDEFINED"
    }

    fun initConfigs()

    fun handle(featuresResponse: JSONObject)
}

object DevicePerformanceClassConfigImpl : DevicePerformanceClassConfig {

    private var storedDevicePerformanceClass
            by corePref(DEVICE_PERFORMANCE_CLASS_VALUE, PERFORMANCE_CLASS_UNDEFINED)

    private var lowDevicesTrimmingPercentage
            by corePref(LOW_TRIMMING_PERCENTAGE, DEFAULT_TRIMMING_PERCENTAGE)

    private var averageDevicesTrimmingPercentage
            by corePref(AVERAGE_TRIMMING_PERCENTAGE, DEFAULT_TRIMMING_PERCENTAGE)

    override fun initConfigs() {
        val isFeatureEnabled = SettingsManager.getInstance()
            .isFeatureEnabled(IBGFeature.DEVICE_PERFORMANCE_CLASS, false)

        if (!isFeatureEnabled) {
            clearSavedValues()
            return
        }

        if (storedDevicePerformanceClass > PERFORMANCE_CLASS_UNDEFINED) {
            setDevicePerformanceClassValue(storedDevicePerformanceClass)
        }

        if (!overrideDevicePerformanceClass() && Instabug.getApplicationContext() != null) {
            DevicePerformanceClassUtils(Instabug.getApplicationContext()!!).run {
                setDevicePerformanceClassValue(measureDevicePerformanceClass())
            }
        }

        setDeviceTrimmingPercentage(
            storedDevicePerformanceClass,
            lowDevicesTrimmingPercentage,
            averageDevicesTrimmingPercentage
        )
    }

    override fun handle(featuresResponse: JSONObject) {
        val diagnosticsObject = featuresResponse.optJSONObject("diagnostics")
        val dvPerfObject = diagnosticsObject?.optJSONObject("dv_perf")

        val featurePercentage = dvPerfObject?.optDouble("dv_perf_class") ?: 0.0
        IBGPercentageFlagsResolver.resolve(IBGFeature.DEVICE_PERFORMANCE_CLASS, featurePercentage)
        val isFeatureEnabled = SettingsManager.getInstance()
            .isFeatureEnabled(IBGFeature.DEVICE_PERFORMANCE_CLASS, false)

        if (!isFeatureEnabled) {
            clearSavedValues()
            return
        }

        val devicesPerformanceClass = dvPerfObject?.optJSONObject("dv_perf_class_override")

        devicesPerformanceClass?.optJSONArray("low_perf_device")?.let {
            storeDevicesOverride(LOW_DEVICES_CLASS_LIST_KEY, parseDevicesOverrideResponse(it))
        } ?: storeDevicesOverride(LOW_DEVICES_CLASS_LIST_KEY, emptySet())

        devicesPerformanceClass?.optJSONArray("average_perf_device")?.let {
            storeDevicesOverride(AVERAGE_DEVICES_CLASS_LIST_KEY, parseDevicesOverrideResponse(it))
        } ?: storeDevicesOverride(AVERAGE_DEVICES_CLASS_LIST_KEY, emptySet())

        devicesPerformanceClass?.optJSONArray("high_perf_device")?.let {
            storeDevicesOverride(HIGH_DEVICES_CLASS_LIST_KEY, parseDevicesOverrideResponse(it))
        } ?: storeDevicesOverride(HIGH_DEVICES_CLASS_LIST_KEY, emptySet())

        if (!overrideDevicePerformanceClass() &&
            Instabug.getApplicationContext() != null
        ) DevicePerformanceClassUtils(Instabug.getApplicationContext()!!).run {
            setDevicePerformanceClassValue(measureDevicePerformanceClass())
        }

        handleTrimmingPercentages(dvPerfObject)
    }

    private fun handleTrimmingPercentages(dvPerfObject: JSONObject?) {
        dvPerfObject?.optDouble(PERFORMANCE_CLASS_LOW_LIMIT_FLAG)?.let {
            lowDevicesTrimmingPercentage = if (it > 0 && it <= 1) {
                it.toFloat()
            } else {
                1.0F
            }
        }

        dvPerfObject?.optDouble(PERFORMANCE_CLASS_AVG_LIMIT_FLAG)?.let {
            averageDevicesTrimmingPercentage = if (it > 0 && it <= 1) {
                it.toFloat()
            } else {
                1.0F
            }
        }

        setDeviceTrimmingPercentage(
            storedDevicePerformanceClass,
            lowDevicesTrimmingPercentage,
            averageDevicesTrimmingPercentage
        )
    }

    private fun setDeviceTrimmingPercentage(
        deviceClass: Int,
        lowTrimmingPercentage: Float,
        averageTrimmingPercentage: Float
    ) {
        val deviceTrimmingPercentage = measureDeviceTrimmingPercentage(
            deviceClass,
            lowTrimmingPercentage,
            averageTrimmingPercentage
        )

        PerSessionSettings.getInstance().deviceTrimmingPercentage = deviceTrimmingPercentage

        logTrimmingPercentageForDevice(storedDevicePerformanceClass, deviceTrimmingPercentage)
    }

    private fun measureDeviceTrimmingPercentage(
        deviceClass: Int,
        lowTrimmingPercentage: Float,
        averageTrimmingPercentage: Float
    ): Float {
        return when (deviceClass) {
            PERFORMANCE_CLASS_LOW -> lowTrimmingPercentage
            PERFORMANCE_CLASS_AVERAGE -> averageTrimmingPercentage
            else -> 1.0F
        }
    }

    private fun getDevicePerformanceClassFromOverride(): Int {
        if (Instabug.getApplicationContext() != null) {
            val devicePerformanceClassUtils =
                DevicePerformanceClassUtils(Instabug.getApplicationContext()!!)
            when {
                devicePerformanceClassUtils.isDevicePerformanceClassMatched(
                    getStoredDevicesOverride(LOW_DEVICES_CLASS_LIST_KEY)
                ) -> return PERFORMANCE_CLASS_LOW

                devicePerformanceClassUtils.isDevicePerformanceClassMatched(
                    getStoredDevicesOverride(AVERAGE_DEVICES_CLASS_LIST_KEY)
                ) -> return PERFORMANCE_CLASS_AVERAGE

                devicePerformanceClassUtils.isDevicePerformanceClassMatched(
                    getStoredDevicesOverride(HIGH_DEVICES_CLASS_LIST_KEY)
                ) -> return PERFORMANCE_CLASS_HIGH
            }
        }

        return PERFORMANCE_CLASS_UNDEFINED
    }

    /**
     * Override the current identified device performance class from an available override lists.
     *
     * @return true if the device hash matched any hash in override list and new performance class
     * take a place or false otherwise.
     */
    @VisibleForTesting
    fun overrideDevicePerformanceClass(): Boolean {
        val overriddenPerformanceClass = getDevicePerformanceClassFromOverride()
        if (overriddenPerformanceClass > PERFORMANCE_CLASS_UNDEFINED) {
            setDevicePerformanceClassValue(overriddenPerformanceClass)
            logDevicePerformanceClassOverridden(overriddenPerformanceClass)

            return true
        }
        return false
    }

    @VisibleForTesting
    fun storeDevicesOverride(performanceClass: String, devicesOverrideList: Set<Int>) {
        var presidedDevices by corePref(performanceClass, setOf<Int>())
        presidedDevices = devicesOverrideList
    }

    @VisibleForTesting
    fun getStoredDevicesOverride(devicesClassKey: String): Set<Int> {
        val presidedDevices by corePref(devicesClassKey, setOf<Int>())
        return presidedDevices
    }

    private fun clearSavedValues() {
        setDevicePerformanceClassValue(PERFORMANCE_CLASS_UNDEFINED)
        PersistableSettings.getInstance()?.clearValue(DEVICE_PERFORMANCE_CLASS_VALUE)
        clearStoredDevicesOverrideList()
        clearStoredTrimmingPercentages()
        PersistableSettings.getInstance()
            ?.clearPercentageFeature(IBGFeature.DEVICE_PERFORMANCE_CLASS)
    }

    @VisibleForTesting
    fun clearStoredDevicesOverrideList() {
        PersistableSettings.getInstance()?.clearValue(LOW_DEVICES_CLASS_LIST_KEY)
        PersistableSettings.getInstance()?.clearValue(AVERAGE_DEVICES_CLASS_LIST_KEY)
        PersistableSettings.getInstance()?.clearValue(HIGH_DEVICES_CLASS_LIST_KEY)
    }

    private fun clearStoredTrimmingPercentages() {
        PersistableSettings.getInstance()?.clearValue(LOW_TRIMMING_PERCENTAGE)
        PersistableSettings.getInstance()?.clearValue(AVERAGE_TRIMMING_PERCENTAGE)
        PerSessionSettings.getInstance().deviceTrimmingPercentage = 1.0F
    }

    @VisibleForTesting
    fun parseDevicesOverrideResponse(devices: JSONArray): Set<Int> {
        val devicesSet = mutableSetOf<Int>()

        for (index in 0 until devices.length())
            devicesSet.add(devices.getInt(index))

        return devicesSet
    }

    private fun setDevicePerformanceClassValue(value: Int) {
        SettingsManager.getInstance().devicePerformanceClass = value
        storedDevicePerformanceClass = value
    }

    private fun logTrimmingPercentageForDevice(
        devicePerformanceClass: Int,
        deviceTrimmingPercentage: Float
    ) {
        val devicePerformanceClassLiteral = getDevicePerformanceClassLiteral(devicePerformanceClass)

        InstabugSDKLogger.w(
            Constants.LOG_TAG, "Device has been classified as $devicePerformanceClassLiteral, " +
                    "${(deviceTrimmingPercentage * 100).toInt()}% of logs are being saved."
        )
    }

    private fun logDevicePerformanceClassOverridden(devicePerformanceClass: Int) {
        val devicePerformanceClassLiteral = getDevicePerformanceClassLiteral(devicePerformanceClass)
        InstabugSDKLogger.v(
            Constants.LOG_TAG,
            "Device class value has been overridden, Device class: $devicePerformanceClassLiteral"
        )
    }

    private fun getDevicePerformanceClassLiteral(devicePerformanceClass: Int) =
        when (devicePerformanceClass) {
            PERFORMANCE_CLASS_LOW -> LOW
            PERFORMANCE_CLASS_AVERAGE -> AVERAGE
            PERFORMANCE_CLASS_HIGH -> HIGH
            else -> UNDEFINED
        }

}