package com.instabug.library.model.session;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_APP_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_APP_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_CRASH_REPORTING_ENABLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_DEVICE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_DURATION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_OS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_PRODUCTION_USAGE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_SDK_VERSION;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_STARTED_AT;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_STITCHED_SESSION_LEAD;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_SYNC_STATUS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USERS_PAGE_ENABLED;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_ATTRIBUTES;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_ATTRIBUTES_KEYS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_EMAIL;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_EVENTS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_EVENTS_KEYS;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_USER_NAME;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_UUID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.SessionEntry.COLUMN_V2_SESSION_SENT;
import static com.instabug.library.model.session.SessionParameter.APP_VERSION;
import static com.instabug.library.model.session.SessionParameter.CRASH_REPORTING_ENABLED;
import static com.instabug.library.model.session.SessionParameter.CUSTOM_ATTRIBUTES;
import static com.instabug.library.model.session.SessionParameter.CUSTOM_ATTRIBUTES_KEYS;
import static com.instabug.library.model.session.SessionParameter.DEVICE;
import static com.instabug.library.model.session.SessionParameter.DURATION;
import static com.instabug.library.model.session.SessionParameter.OS;
import static com.instabug.library.model.session.SessionParameter.SDK_VERSION;
import static com.instabug.library.model.session.SessionParameter.STARTED_AT;
import static com.instabug.library.model.session.SessionParameter.STITCHED_SESSION_LEAD;
import static com.instabug.library.model.session.SessionParameter.USER_EMAIL;
import static com.instabug.library.model.session.SessionParameter.USER_EVENTS;
import static com.instabug.library.model.session.SessionParameter.USER_EVENTS_KEYS;
import static com.instabug.library.model.session.SessionParameter.USER_NAME;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.database.Cursor;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.internal.utils.stability.handler.exception.ExceptionHandler;
import com.instabug.library.networkv2.request.Endpoints;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestMethod;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

@Keep
public class SessionMapper {

    private static final Object TAG = "SessionMapper";

    private SessionMapper() {
        // Utility classes should not be instantiable
    }

    @NonNull
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static SessionRemoteEntity toRemoteEntity(@NonNull CoreSession session) {
        Map<String, Object> map = new HashMap<>();
        map.put(OS, session.getOs());
        map.put(DEVICE, session.getDevice());
        map.put(DURATION, session.getDuration());
        map.put(STARTED_AT, session.getStartTimestampMicros());
        map.put(USER_NAME, session.getUserName());
        map.put(USER_EMAIL, session.getUserEmail());
        map.put(SDK_VERSION, session.getSdkVersion());
        map.put(APP_VERSION, session.getAppVersion());
        map.put(session.isUsersPageEnabled() ? USER_EVENTS : USER_EVENTS_KEYS, session.getUserEvents());
        map.put(session.isUsersPageEnabled() ? CUSTOM_ATTRIBUTES : CUSTOM_ATTRIBUTES_KEYS, session.getCustomAttributes());
        map.put(CRASH_REPORTING_ENABLED, session.isCrashReportingEnabled());
        map.put(Constants.NetworkParam.UUID, session.getUuid());
        map.put(Constants.NetworkParam.APP_TOKEN, session.getAppToken());
        map.put(STITCHED_SESSION_LEAD, session.isStitchedSessionLead());
        return toRemoteEntity(session.getId(), map);
    }

    @NonNull
    public static SessionRemoteEntity toRemoteEntity(String id, @NonNull Map<String, Object> map) {
        return new SessionRemoteEntity(id, map);
    }

    @NonNull
    public static SessionsBatchDTO toDTO(@NonNull SessionRemoteEntity session) {
        Map<String, Object> commonKeys = Collections.emptyMap();
        List<SessionRemoteEntity> sessions = Collections.singletonList(session);
        return toDTO(commonKeys, sessions);
    }

    @NonNull
    public static SessionsBatchDTO toDTO(@NonNull Map<String, Object> commonKeys, @NonNull List<SessionRemoteEntity> sessions) {
        return new SessionsBatchDTO(commonKeys, sessions);
    }

    @NonNull
    public static CoreSession toModel(@NonNull SessionLocalEntity entity) {
        return new CoreSession.Builder(entity.getId(), entity.getUuid(), entity.getOs())
                .setDevice(entity.getDevice())
                .setDuration(entity.getDuration())
                .setStartTimestampMicros(entity.getStartTimestampMicros())
                .setUserName(entity.getUserName())
                .setUserEmail(entity.getUserEmail())
                .setSdkVersion(entity.getSdkVersion())
                .setAppVersion(entity.getAppVersion())
                .setUserEvents(entity.isUsersPageEnabled() ? entity.getUserEvents() : entity.getUserEventsKeys())
                .setCustomAttributes(entity.isUsersPageEnabled() ? entity.getCustomAttributes() : entity.getCustomAttributesKeys())
                .setCrashReportingEnabled(entity.isCrashReportingEnabled())
                .setSyncStatus(entity.getSyncStatus())
                .setAppToken(entity.getAppToken())
                .setUsersPageEnabled(entity.isUsersPageEnabled())
                .setProductionUsage(entity.getProductionUsage())
                .setIsStitchedSessionLead(entity.isStitchedSessionLead())
                .build();
    }

    @NonNull
    public static List<CoreSession> toModels(@NonNull List<SessionLocalEntity> entities) {
        List<CoreSession> sessions = new ArrayList<>();
        for (SessionLocalEntity entity : entities) {
            sessions.add(toModel(entity));
        }
        return sessions;
    }

    @NonNull
    public static ContentValues toContentValues(SessionLocalEntity entity) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_ID, entity.getId());
        contentValues.put(COLUMN_STARTED_AT, entity.getStartTimestampMicros());
        contentValues.put(COLUMN_DURATION, entity.getDuration());
        contentValues.put(COLUMN_USER_EVENTS, entity.getUserEvents());
        contentValues.put(COLUMN_USER_ATTRIBUTES, entity.getCustomAttributes());
        contentValues.put(COLUMN_USER_EVENTS_KEYS, entity.getUserEventsKeys());
        contentValues.put(COLUMN_USER_ATTRIBUTES_KEYS, entity.getCustomAttributesKeys());
        contentValues.put(COLUMN_USER_EMAIL, entity.getUserEmail());
        contentValues.put(COLUMN_USER_NAME, entity.getUserName());
        contentValues.put(COLUMN_UUID, entity.getUuid());
        contentValues.put(COLUMN_APP_TOKEN, entity.getAppToken());
        contentValues.put(COLUMN_OS, entity.getOs());
        contentValues.put(COLUMN_DEVICE, entity.getDevice());
        contentValues.put(COLUMN_SDK_VERSION, entity.getSdkVersion());
        contentValues.put(COLUMN_APP_VERSION, entity.getAppVersion());
        contentValues.put(COLUMN_CRASH_REPORTING_ENABLED, entity.isCrashReportingEnabled() ? 1 : 0);
        contentValues.put(COLUMN_USERS_PAGE_ENABLED, entity.isUsersPageEnabled() ? 1 : 0);
        contentValues.put(COLUMN_SYNC_STATUS, entity.getSyncStatus());
        contentValues.put(COLUMN_STITCHED_SESSION_LEAD, entity.isStitchedSessionLead() ? 1 : 0);
        contentValues.put(COLUMN_V2_SESSION_SENT, entity.isSessionV2Sent() ? 1 : 0);
        String productionUsage = entity.getProductionUsage();
        if (productionUsage != null) {
            contentValues.put(COLUMN_PRODUCTION_USAGE, productionUsage);
        }
        return contentValues;
    }

    @NonNull
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static JSONObject toJson(@NonNull final SessionRemoteEntity entity) {
        return new ExceptionHandler()
                .withPenaltyLog()
                .executeAndGet(() -> {
                    JSONObject jsonObject = new JSONObject();
                    for (Map.Entry<String, Object> keyValueEntry : entity.getMap().entrySet()) {
                        // Don't attach 'stitched_session_lead' to sessions sync request if feature is disabled
                        if (!InstabugFeaturesManager.getInstance().isSessionStitchingEnabled()
                                && keyValueEntry.getKey() != null && keyValueEntry.getKey().equals(STITCHED_SESSION_LEAD)) {
                            continue;
                        }
                        jsonObject.put(keyValueEntry.getKey(), extractJsonValue(keyValueEntry.getValue()));
                    }
                    return jsonObject;
                }, new JSONObject());
    }

    @NonNull
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static JSONObject toJson(@NonNull final SessionsBatchDTO batch) {
        return new ExceptionHandler()
                .withPenaltyLog()
                .executeAndGet(() -> {
                    JSONObject jsonObject = new JSONObject();
                    String productionUsage = batch.getProductionUsage();
                    if (productionUsage != null && !productionUsage.isEmpty()) {
                        try {
                            jsonObject.put("production_usage", new JSONObject(productionUsage));
                        } catch (JSONException e) {
                            InstabugSDKLogger.e(Constants.LOG_TAG, e.getClass().getSimpleName(), e);
                        }
                    }
                    // populate common keys to main object
                    for (Map.Entry<String, Object> keyValueEntry : batch.getCommonKeys().entrySet()) {
                        jsonObject.put(keyValueEntry.getKey(), extractJsonValue(keyValueEntry.getValue()));
                    }
                    JSONArray jsonArray = new JSONArray();
                    // populate sessions to inner array
                    for (SessionRemoteEntity session : batch.getSessions()) {
                        jsonArray.put(toJson(session));
                    }
                    // populate sessions array to main object
                    jsonObject.put("sessions", jsonArray);
                    return jsonObject;
                }, new JSONObject());
    }

    @NonNull
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static Request toRequest(@NonNull final JSONObject batch) {
        final Request.Builder requestBuilder = new Request.Builder()
                .endpoint(Endpoints.SEND_SESSION)
                .method(RequestMethod.POST);
        return new ExceptionHandler()
                .withPenaltyLog()
                .executeAndGet(() -> {
                    Iterator<String> keys = batch.keys();
                    while (keys.hasNext()) {
                        String key = keys.next();
                        requestBuilder.addParameter(new RequestParameter(key, batch.get(key)));
                    }
                    return requestBuilder.build();
                }, requestBuilder.build());
    }

    @NonNull
    public static SessionLocalEntity toLocalEntity(@NonNull Cursor cursor) {
        String id = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_ID));
        long startedAt = cursor.getLong(cursor.getColumnIndexOrThrow(COLUMN_STARTED_AT));
        long duration = cursor.getLong(cursor.getColumnIndexOrThrow(COLUMN_DURATION));
        String userEvents = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_USER_EVENTS));
        String userAttributes = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_USER_ATTRIBUTES));
        String userEventsKeys = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_USER_EVENTS_KEYS));
        String userAttributesKeys = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_USER_ATTRIBUTES_KEYS));
        String userEmail = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_USER_EMAIL));
        String userName = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_USER_NAME));
        String uuid = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_UUID));
        String appToken = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_APP_TOKEN));
        String os = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_OS));
        String device = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_DEVICE));
        String sdkVersion = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_SDK_VERSION));
        String appVersion = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_APP_VERSION));
        int crashReportingEnabled = cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_CRASH_REPORTING_ENABLED));
        int syncStatus = cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_SYNC_STATUS));
        int userPageEnabled = cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_USERS_PAGE_ENABLED));
        String productionUsage = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_PRODUCTION_USAGE));
        int isSessionLead = cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_STITCHED_SESSION_LEAD));
        int isSessionV2Sent = cursor.getInt(cursor.getColumnIndexOrThrow(COLUMN_V2_SESSION_SENT));

        return new SessionLocalEntity(
                id,
                os,
                device,
                duration,
                startedAt,
                userName,
                userEmail,
                sdkVersion,
                appVersion,
                userEvents,
                userAttributes,
                userEventsKeys,
                userAttributesKeys,
                crashReportingEnabled == 1,
                syncStatus,
                uuid,
                appToken,
                userPageEnabled == 1,
                productionUsage,
                isSessionLead == 1,
                isSessionV2Sent == 1
        );
    }

    @NonNull
    public static List<String> toIDs(@NonNull SessionsBatchDTO batch) {
        List<String> ids = new ArrayList<>();
        for (SessionRemoteEntity session : batch.getSessions()) {
            ids.add(session.getId());
        }
        return ids;
    }

    @NonNull
    public static CoreSession toSession(@NonNull String id, @NonNull String os, @NonNull String uuid,
                                        @Nullable String appVersion, long startTimestampMicros, long startNanoTime, boolean isStitchedSessionLead) {
        return new CoreSession.Builder(id, uuid, os)
                .setAppVersion(appVersion)
                .setStartTimestampMicros(startTimestampMicros)
                .setStartNanoTime(startNanoTime)
                .setIsStitchedSessionLead(isStitchedSessionLead)
                .build();
    }

    private static Object extractJsonValue(Object value) {
        if (value != null && (value.toString().contains("{") || value.toString().contains("["))) {
            try {
                return new JSONObject(value.toString());
            } catch (JSONException e) {
                try {
                    return new JSONArray(value.toString());
                } catch (JSONException ex) {
                    // swallow
                }
            }
        }
        return value;
    }
}
