package com.instabug.library.logging.listeners.networklogs

import com.instabug.library.model.NetworkLog
import com.instabug.library.util.extenstions.toJsonObject
import com.instabug.library.util.extenstions.toMap
import org.json.JSONObject

private const val NULL = "null"

object NetworkLogSnapshotHelper {

    /**
     * Construct a new NetworkLogSnapshot form the captured NetworkLog data.
     *
     * @param networkLog object that contains the original captured networkLog data that can be
     * used to construct and populate the {@link NetworkLogSnapshot} object.
     *
     * @return a simple version of the captured NetworkLog wrapped in a {@link NetworkLogSnapshot} object.
     */
    fun getNetworkLogSnapshot(networkLog: NetworkLog): NetworkLogSnapshot {
        val requestHeaders = networkLog.requestHeaders?.toJsonObjectOrNull()?.toMap()
        val responseHeaders = networkLog.responseHeaders?.toJsonObjectOrNull()?.toMap()
        return NetworkLogSnapshot(
            networkLog.url,
            requestHeaders,
            networkLog.request,
            responseHeaders,
            networkLog.response,
            networkLog.responseCode
        )
    }

    private fun String?.toJsonObjectOrNull(): JSONObject? = this
        ?.takeUnless { NULL == this }
        ?.runCatching { JSONObject(this) }
        ?.getOrNull()


    /**
     * Apply user changes on {@link NetworkLogSnapshot} to the captured {@link NetworkLog}.
     *
     * @param networkLog object that contains the original captured networkLog data. This object is
     * also used to apply the data changes on it before it got inserted in the database.
     * @param userNetworkLogSnapshot object that user use to view and modify the captured NetworkLog.
     */
    fun applyNetworkLogChanges(
        networkLog: NetworkLog, userNetworkLogSnapshot: NetworkLogSnapshot?
    ) {
        if (userNetworkLogSnapshot == null) {
            return
        } else if (userNetworkLogSnapshot != getNetworkLogSnapshot(networkLog)) {
            val snapshotRequestHeaders =
                userNetworkLogSnapshot.requestHeaders?.toJsonObject()?.toString()
            val snapshotResponseHeaders =
                userNetworkLogSnapshot.responseHeaders?.toJsonObject()?.toString()

            networkLog.isUserModified = true
            networkLog.url = userNetworkLogSnapshot.url
            networkLog.requestHeaders = snapshotRequestHeaders
            networkLog.request = userNetworkLogSnapshot.requestBody
            networkLog.responseHeaders = snapshotResponseHeaders
            networkLog.response = userNetworkLogSnapshot.response
        }

        networkLog.insert()
    }

}