package com.instabug.library.logging.disklogs

import android.content.Context
import com.instabug.library.InstabugState
import com.instabug.library.InstabugStateProvider
import com.instabug.library.internal.dataretention.files.FileInspector
import com.instabug.library.internal.orchestrator.DisposeDataAction
import com.instabug.library.internal.storage.DiskUtils
import java.lang.ref.WeakReference
import java.util.concurrent.Executor

interface FileDiskOperator {
    fun flush(encrypted: Boolean, logsStringBuilder: StringBuilder)
    fun clear()
}

class LogsFileDiskOperator(
    private val executor: Executor,
    private val loggingFileProvider: LoggingFileProvider,
    context: Context?,
) : FileDiskOperator {

    private val contextReference: WeakReference<Context> = WeakReference(context)

    override fun flush(encrypted: Boolean, logsStringBuilder: StringBuilder) {
        executor.execute {
            writeLogs(encrypted, logsStringBuilder)
        }
    }

    override fun clear() {
        executor.execute {
            DisposeDataAction(FileInspector.create()).run()
        }
    }

    private fun writeLogs(encrypted: Boolean, logsStringBuilder: StringBuilder) {
        if (InstabugStateProvider.getInstance().state != InstabugState.DISABLED) {
            val todayFile = loggingFileProvider.getTodayFile()
            val context = contextReference.get()
            if (todayFile != null && context != null) {
                DiskUtils.with(context).writeOperation(
                    WriteLogDiskOperator(
                        todayFile,
                        logsStringBuilder.toString(),
                        encrypted
                    )
                ).execute()
                logsStringBuilder.setLength(0)
                loggingFileProvider.trimDirectoryIfNeeded()
            }
        } else {
            logsStringBuilder.setLength(0)
        }
    }
}
