package com.instabug.library.internal.video;

import static com.instabug.library.util.FileUtils.isVideoFile;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.media.MediaCodec;
import android.media.MediaExtractor;
import android.media.MediaFormat;
import android.media.MediaMetadataRetriever;
import android.media.MediaMuxer;
import android.os.Build;
import android.text.TextUtils;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.HashMap;


/**
 * Created by mNagy on 1/16/17.
 */

public class InstabugVideoUtils {

    private static final int DEFAULT_BUFFER_SIZE = 4096;

    /**
     * Getting the duration of a video file
     */
    @SuppressLint("RESOURCE_LEAK")
    private static int getVideoDuration(final String srcPath) {
        MediaMetadataRetriever retriever = new MediaMetadataRetriever();
        int videoDuration = 0;
        try {
            retriever.setDataSource(srcPath);
            String time = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
            if (!TextUtils.isEmpty(time)) {
                videoDuration = Integer.parseInt(time);
            }
        } catch (Exception exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, String.format("Error: %s occurred during getting video file with path: %s duration",
                    exception.getMessage(), srcPath));
        } finally {
            try {
                retriever.release();
            } catch (IOException e) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Error while releasing mediaMetadataRetriever" + e);
            }
        }
        return videoDuration;
    }

    private static boolean isDurationMoreThanAutoRecMaxDuration(int durationMs, int
            autoRecordingMaxDuration) {
        return durationMs > autoRecordingMaxDuration;
    }

    /**
     * Shortens/Crops tracks
     */
    public static File startTrim(File src, File dst, int autoRecordingMaxDuration)
            throws IOException {
        if (src != null && src.length() > 0 && isVideoFile(src)) {
            int videoDuration = getVideoDuration(src.getPath());
            if (videoDuration != 0) {
                if (isDurationMoreThanAutoRecMaxDuration(videoDuration,
                        autoRecordingMaxDuration)) {

                    int startMs = videoDuration - autoRecordingMaxDuration;
                    genVideoUsingMuxer(src.getPath(), dst.getPath(), startMs, videoDuration, false, true);
                    return dst;
                }
            }
        }
        return src;
    }
    /**
     * @param srcPath  the path of source video file.
     * @param dstPath  the path of destination video file.
     * @param startMs  starting time in milliseconds for trimming. Set to
     *                 negative if starting from beginning.
     * @param endMs    end time for trimming in milliseconds. Set to negative if
     *                 no trimming at the end.
     * @param useAudio true if keep the audio track from the source.
     * @param useVideo true if keep the video track from the source.
     * @throws IOException
     */
    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    @SuppressLint("RESOURCE_LEAK")
    private static void genVideoUsingMuxer(String srcPath, String dstPath,
                                           int startMs, int endMs, boolean useAudio, boolean useVideo)
            throws IOException {
        // Set up MediaExtractor to read from the source.
        MediaExtractor extractor = new MediaExtractor();
        extractor.setDataSource(srcPath);
        int trackCount = extractor.getTrackCount();
        // Set up MediaMuxer for the destination.
        MediaMuxer muxer;
        muxer = new MediaMuxer(dstPath, MediaMuxer.OutputFormat.MUXER_OUTPUT_MPEG_4);
        // Set up the tracks and retrieve the max buffer size for selected
        // tracks.
        HashMap<Integer, Integer> indexMap = new HashMap<>(trackCount);
        int bufferSize = -1;
        for (int i = 0; i < trackCount; i++) {
            MediaFormat format = extractor.getTrackFormat(i);
            String mime = format.getString(MediaFormat.KEY_MIME);
            boolean selectCurrentTrack = false;
            if (mime != null && mime.startsWith("audio/") && useAudio) {
                selectCurrentTrack = true;
            } else if (mime != null && mime.startsWith("video/") && useVideo) {
                selectCurrentTrack = true;
            }
            if (selectCurrentTrack) {
                extractor.selectTrack(i);
                int dstIndex = muxer.addTrack(format);
                indexMap.put(i, dstIndex);
                if (format.containsKey(MediaFormat.KEY_MAX_INPUT_SIZE)) {
                    int newSize = format.getInteger(MediaFormat.KEY_MAX_INPUT_SIZE);
                    bufferSize = newSize > bufferSize ? newSize : bufferSize;
                }
            }
        }
        if (bufferSize < 0) {
            bufferSize = DEFAULT_BUFFER_SIZE;
        }
        // Set up the orientation and starting time for extractor.
        MediaMetadataRetriever retrieverSrc = new MediaMetadataRetriever();
        retrieverSrc.setDataSource(srcPath);
        String degreesString = retrieverSrc.extractMetadata(
                MediaMetadataRetriever.METADATA_KEY_VIDEO_ROTATION);
        retrieverSrc.release();
        if (degreesString != null) {
            int degrees = Integer.parseInt(degreesString);
            if (degrees >= 0) {
                muxer.setOrientationHint(degrees);
            }
        }
        if (startMs > 0) {
            extractor.seekTo(startMs * 1000, MediaExtractor.SEEK_TO_CLOSEST_SYNC);
        }
        // Copy the samples from MediaExtractor to MediaMuxer. We will loop
        // for copying each sample and stop when we get to the end of the source
        // file or exceed the end time of the trimming.
        int offset = 0;
        int trackIndex = -1;
        boolean isMuxerHasWrittenFrames = false;
        ByteBuffer dstBuf = ByteBuffer.allocate(bufferSize);
        MediaCodec.BufferInfo bufferInfo = new MediaCodec.BufferInfo();
        try {
            muxer.start();
            while (true) {
                bufferInfo.offset = offset;
                bufferInfo.size = extractor.readSampleData(dstBuf, offset);
                if (bufferInfo.size < 0) {
                    bufferInfo.size = 0;
                    break;
                } else {
                    bufferInfo.presentationTimeUs = extractor.getSampleTime();
                    if (endMs > 0 && bufferInfo.presentationTimeUs > (endMs * 1000)) {
                        break;
                    } else {
                        bufferInfo.flags = extractor.getSampleFlags();
                        trackIndex = extractor.getSampleTrackIndex();
                        Integer index = indexMap.get(trackIndex);
                        if (index != null) {
                            muxer.writeSampleData(index, dstBuf,
                                    bufferInfo);
                            isMuxerHasWrittenFrames = true;
                            extractor.advance();
                        }
                    }
                }
            }
            muxer.stop();

            //deleting the old file
            File file = new File(srcPath);
            file.delete();
        } catch (IllegalStateException e) {
            // Swallow the exception due to malformed source.
            InstabugSDKLogger.e(Constants.LOG_TAG, "The source video file is malformed");
        } finally {
            if (isMuxerHasWrittenFrames) {
                muxer.release();
                InstabugSDKLogger.v(Constants.LOG_TAG, "The muxer has been released!");
            } else {
                InstabugSDKLogger.v(Constants.LOG_TAG, "The muxer skip release/stop since it has no written frames");
            }
        }
    }
}
