package com.instabug.library.interactionstracking

import android.content.Context
import android.graphics.Rect
import androidx.annotation.IntDef
import com.instabug.library.model.StepType
import com.instabug.library.model.UserStep
import com.instabug.library.visualusersteps.TouchedView
import java.util.concurrent.Future

/**
 * An representation contract for UI nodes in the screen hierarchy.
 */
interface IBGUINode {
    /**
     * Whether the node represents an IBG View.
     */
    val isIBGView: Boolean

    /**
     * Whether the node represents a visible UI component.
     */
    val isVisible: Boolean

    /**
     * Whether the node represents a clickable UI component.
     */
    val isClickable: Boolean

    /**
     * Whether the node represents a long-clickable UI component.
     */
    val isLongClickable: Boolean

    /**
     * The number of contained children if the node represents a container UI component, 0 otherwise.
     * @see isAContainer
     */
    val childCount: Int

    /**
     * Whether the node represents a container UI component.
     */
    val isAContainer: Boolean

    /**
     * The Z index of the UI component the node represents in the screen hierarchy.
     */
    val zOrder: Float

    /**
     * Whether the node represents a scrollable UI component.
     */
    val isScrollable: Boolean

    /**
     * Whether the node represents a swipeable UI component.
     */
    val isSwipeable: Boolean

    /**
     * The wrapped UI component's class name.
     */
    val componentClassName: String

    /**
     * Whether the node represents a checkable UI component
     */
    val isCheckable: Boolean

    /**
     * Whether the underlying UI component is currently checked.
     * Always returning false if the underlying origin is not checkable.
     * @see isCheckable
     */
    val isChecked: Boolean

    /**
     * Whether the underlying UI component represents some sort of movable progress seeker.
     */
    val isMovableWithProgress: Boolean

    /**
     * Whether the underlying UI component represents a text field of sort.
     */
    val isTextField: Boolean

    /**
     * Whether the underlying UI component is focusable.
     */
    val isFocusable: Boolean

    /**
     * Whether the underlying UI component is a valid tap-like gestures compatible.
     * Defaults to true to be overridden in special cases.
     */
    val isValidTappableTarget: Boolean
        get() = true

    /**
     * Whether the UI component represented by this node is private
     */
    val isPrivate: Boolean

    /**
     * The type of the UI component represented by this node.
     */
    @Type
    val type: Int

    /**
     * Checks whether the given touch co-ordinate (x, y) falls within the bounds of the UI component
     * the node represents.
     * @param x the touch co-ordinate on the x-axis of the screen.
     * @param y the touch co-ordinate on the y-axis of the screen.
     * @return a [Boolean] indicating whether the given co-ordinates falls within bounds.
     */
    fun isTouchWithinBounds(x: Float, y: Float): Boolean

    /**
     * Whether the underlying UI component is a valid touch target.
     * Defaults to the basic conditions of acceptance to be overridden in special cases.
     */
    fun isValidTouchTarget(x: Float, y: Float): Boolean {
        return !isIBGView && isVisible && isTouchWithinBounds(x, y)
    }

    /**
     * Queries the children of the UI component to fetch the child at the given index.
     * The node should be representing a container UI component or the result of using this
     * query is null
     * @param index the index of the child to get from the parent container.
     * @return an [IBGUINode] representing the child UI component or null if the component is not a container
     * or no child found at the given index
     * @see isAContainer
     */
    fun getChildAt(index: Int): IBGUINode?

    /**
     * Generates a repro interaction label representing the interaction done on the UI component.
     * @return a [Future] of [TouchedView] that represents the appropriate label for the interaction.
     */
    fun asTouchedView(): Future<TouchedView?>?

    fun asRect(): Rect?

    /**
     * Generates a [UserStep] representing the interaction done on the UI component.
     * @param gesture the gesture that the user has done on screen.
     * @param holder the context (usually an activity) of lifecycle container that contains this UI component.
     * @return an addable [UserStep] to the sequence of user steps.
     */
    fun asUserStep(@StepType gesture: String, holder: Context): UserStep

    /**
     * Given an array of ids, checks whether the underlying UI component is included in the array.
     * @param ibgComponentsIds the ids representing IBG components
     * @return true if the underlying UI component is included in the array, false otherwise.
     */
    fun isIBGComponent(ibgComponentsIds: IntArray): Boolean

    @IntDef(Type.UNDEFINED, Type.LABEL, Type.INPUT_FIELD, Type.MEDIA)
    @Retention(AnnotationRetention.SOURCE)
    annotation class Type {
        companion object {
            const val UNDEFINED = 0b000
            const val LABEL = 0b001
            const val INPUT_FIELD = 0b010
            const val MEDIA = 0b100
        }
    }
}