package com.instabug.library.diagnostics.customtraces.cache

import com.instabug.library.diagnostics.customtraces.model.IBGCustomTrace

interface CustomTracesCacheManager {

    /**
     * Starts a custom trace
     *
     * @param traceId in the db
     */
    fun startTrace(trace: IBGCustomTrace): Long

    /**
     * ends custom trace
     *
     * @param traceId id of the trace in the db
     * @param duration trace duration
     *
     * @return true if trace was updated in the db successfully
     */
    fun endTrace(traceId: Long, duration: Long, endedInBG: Boolean): Boolean

    /**
     * Adds attribute to trace
     *
     * @param traceId to use as a foreign key
     * @param key attribute key
     * @param value attribute value
     *
     * @return true if attribute was inserted to db
     */
    fun setAttribute(traceId: Long, key: String, value: String?): Boolean

    /**
     * Updates attribute to trace
     *
     * @param traceId to use as a foreign key
     * @param key attribute key
     * @param value attribute value
     *
     * @return true if attribute was inserted to db
     */
    fun updateAttribute(traceId: Long, key: String, value: String?): Boolean

    fun getAllTraces(): List<IBGCustomTrace>

    /**
     * Removes traces whose duration was not set from the db
     */
    fun removeUnEndedTraces()

    /**
     * Clear all traces in the db
     */
    fun deleteAll()

    /**
     * Inserts a trace into db
     *
     * @param traceName identifier
     * @param startTime time when the trace started
     * @param duration trace duration
     * @param isBG true when the session was started and ended in BG
     */
    fun logTrace(traceName: String, startTime: Long, duration: Long, isBG: Boolean)

    /**
     * Gets the id of the trace in the db
     *
     * @param trace trace object
     */
    fun getTraceId(trace: IBGCustomTrace) : Long

    /**
     * Deletes synced traces from the db
     *
     * @param traces synced traces
     */
    fun clearSyncedTraces(traces: List<IBGCustomTrace>)

    fun clearTracesByName(tracesNames: Array<String>)
}