package com.instabug.library.visualusersteps;

import static com.instabug.library.Instabug.getApplicationContext;
import static com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventBusKt.TYPE_SESSION;
import static com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventBusKt.TYPE_V3_SESSION;
import static com.instabug.library.model.StepType.ACTIVITY_CREATED;
import static com.instabug.library.model.StepType.ACTIVITY_DESTROYED;
import static com.instabug.library.model.StepType.ACTIVITY_PAUSED;
import static com.instabug.library.model.StepType.ACTIVITY_RESUMED;
import static com.instabug.library.model.StepType.ACTIVITY_STARTED;
import static com.instabug.library.model.StepType.ACTIVITY_STOPPED;
import static com.instabug.library.model.StepType.APPLICATION_BACKGROUND;
import static com.instabug.library.model.StepType.APPLICATION_CREATED;
import static com.instabug.library.model.StepType.APPLICATION_FOREGROUND;
import static com.instabug.library.model.StepType.COMPOSE_DISPOSED;
import static com.instabug.library.model.StepType.COMPOSE_PAUSED;
import static com.instabug.library.model.StepType.COMPOSE_RESUMED;
import static com.instabug.library.model.StepType.COMPOSE_STARTED;
import static com.instabug.library.model.StepType.COMPOSE_STOPPED;
import static com.instabug.library.model.StepType.DIALOG_FRAGMENT_RESUMED;
import static com.instabug.library.model.StepType.END_EDITING;
import static com.instabug.library.model.StepType.FRAGMENT_ATTACHED;
import static com.instabug.library.model.StepType.FRAGMENT_DETACHED;
import static com.instabug.library.model.StepType.FRAGMENT_PAUSED;
import static com.instabug.library.model.StepType.FRAGMENT_RESUMED;
import static com.instabug.library.model.StepType.FRAGMENT_STARTED;
import static com.instabug.library.model.StepType.FRAGMENT_STOPPED;
import static com.instabug.library.model.StepType.FRAGMENT_VIEW_CREATED;
import static com.instabug.library.model.StepType.FRAGMENT_VISIBILITY_CHANGED;
import static com.instabug.library.model.StepType.OPEN_DIALOG;
import static com.instabug.library.model.StepType.PINCH;
import static com.instabug.library.model.StepType.SCROLL;
import static com.instabug.library.model.StepType.START_EDITING;
import static com.instabug.library.model.StepType.SWIPE;
import static com.instabug.library.model.StepType.TAB_SELECT;
import static com.instabug.library.model.StepType.UNKNOWN;
import static com.instabug.library.screenshot.instacapture.ScreenShotType.VISUAL_USER_STEP_SCREENSHOT;
import static com.instabug.library.visualusersteps.ParentExtKt.getLastStepType;
import static com.instabug.library.visualusersteps.VisualUserStep.Builder;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.SystemClock;
import android.system.ErrnoException;
import android.system.OsConstants;
import android.text.TextUtils;
import android.util.Pair;
import android.view.View;
import android.widget.EditText;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.fragment.app.Fragment;

import com.instabug.library.Constants;
import com.instabug.library.IBGFeature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.Platform;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventSubscriber;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Session.SessionFinished;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.interactionstracking.IBGUINode;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.utils.memory.IBGLowMemroyWarning;
import com.instabug.library.model.StepType;
import com.instabug.library.screenshot.ScreenshotCaptor;
import com.instabug.library.screenshot.analytics.AnalyticsEvent;
import com.instabug.library.screenshot.analytics.ScreenshotsErrorCodes;
import com.instabug.library.screenshot.instacapture.ScreenshotRequest;
import com.instabug.library.screenshot.instacapture.ScreenshotRequestArgs;
import com.instabug.library.screenshot.subscribers.ScreenshotsAnalyticsEventBus;
import com.instabug.library.sessionreplay.SRScreenshotStore;
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator;
import com.instabug.library.sessionreplay.model.SRScreenshotLog;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.ActivitySecureFlagDetector;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;
import com.instabug.library.util.extenstions.FileExtKt;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.util.threading.SimpleCompletableFuture;

import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;

/**
 * @author hossam.
 */

public class VisualUserStepsProvider implements ReproStepsCaptor {

    @Nullable
    private @StepType String lastConsideredStepType;
    public static final String STEPS_DIRECTORY_NAME = "vusf";
    public static final String STEPS_COMPRESSED_FILE_NAME = "usersteps_";
    static final String DEFAULT_TEXT_FIELD = "a text field";
    @VisibleForTesting
    public static final int VISUAL_USER_STEPS_SCREENSHOTS_LIMIT = 20;
    public static final int VISUAL_USER_STEPS_EVENTS_LIMIT = 100;
    private static final int VISUAL_USER_STEPS_EVENTS_TOLERANCE_RATE = 10;
    private static final String STEP_FILE_PREFIX = "step";
    private static final int SCREENSHOT_QUALITY_PERCENTAGE = 70;
    private static final String SR_REPRO_EXEC_Q_ID = "sr-repro-integration-exec";

    @Nullable
    private static VisualUserStepsProvider INSTANCE;

    @Nullable
    private WeakReference<View> lastView;
    @VisibleForTesting
    VisualUserSteps visualUserSteps;

    @Nullable
    private VisualUserStep pendingForeground;
    private int screenId = 0;
    @Nullable
    private String lastScreenName;
    private boolean shouldCaptureNextAppForeground = true;
    private long lastDialogFragmentDisplayedAt;

    private int platform;
    private long lastScreenshotCapturedAt;
    private Executor executor;

    private final Deque<Pair<Parent, SRScreenshotLog>> pendingComposeSteps = new LinkedList<>();

    @SuppressLint("CheckResult")
    private VisualUserStepsProvider(Executor executor) {
        this.executor = executor;
        platform = SettingsManager.getInstance().getCurrentPlatform();

        visualUserSteps = new VisualUserSteps();
        final Context applicationContext = getApplicationContext();

        if (applicationContext != null) {
            // These operations are kept for compatibility with older SDK releases.
            PoolProvider.postIOTask(() -> {
                File externalDir = VisualUserStepsHelper.getVisualUserStepsDirectory(applicationContext, false);
                if (externalDir != null)
                    FileExtKt.deleteRecursivelyDefensive(externalDir);
                FileExtKt.deleteRecursivelyDefensive(VisualUserStepsHelper.getVisualUserStepsInternalDirectory(applicationContext));
            });
        }
        IBGCoreEventSubscriber.subscribe(ibgSdkCoreEvent -> {
            if (ibgSdkCoreEvent.getType().equals(TYPE_SESSION)) {
                if (ibgSdkCoreEvent instanceof SessionFinished) handleSessionFinished();
                else CoreServiceLocator.getReproStepsProxy().logForegroundStep();
            } else if (ibgSdkCoreEvent.getType().equals(TYPE_V3_SESSION)) {
                if (ibgSdkCoreEvent instanceof IBGSdkCoreEvent.V3Session.V3SessionFinished) {
                    handleSessionFinished();
                } else CoreServiceLocator.getReproStepsProxy().logForegroundStep();

            }
        });
    }

    private void handleSessionFinished() {
        if (!SettingsManager.getInstance().isCrashedSession()) {
            CoreServiceLocator.getReproStepsProxy().logBackgroundStep();
            executor.execute(this::trim);
        }
    }

    public static synchronized VisualUserStepsProvider getInstance(Executor executor) {
        if (INSTANCE == null) {
            INSTANCE = new VisualUserStepsProvider(executor);
        }
        return INSTANCE;
    }

    @Override
    public void setLastView(@Nullable WeakReference<View> viewRef) {
        lastView = viewRef;
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    static void release() {
        INSTANCE = null;
    }

    @Nullable
    @Override
    public Parent getCurrentParent() {
        return visualUserSteps.getLastParent();
    }

    @Nullable
    private Parent getLastParent() {
        if (visualUserSteps.getParents() == null) return null;
        Deque<Parent> parents = visualUserSteps.getParents();
        return parents.peekLast();
    }

    /**
     * This method to add visual user step directly disregarding step type, this api used for RN component.
     *
     * @param screenName
     * @param bitmap
     */
    @Override
    public void addVisualUserStep(String screenName, @Nullable Bitmap bitmap) {
        executor.execute(() -> {
            try {
                // Avoid adding new steps coming from the app activities while the user is taking
                // an extra screenshot or recoding a video
                if (InstabugCore.isForegroundBusy()) {
                    return;
                }

                trim();

                if (screenName != null && !screenName.isEmpty()) {
                    addParent(screenName, ACTIVITY_RESUMED);
                }

                if (!CoreServiceLocator.getReproScreenshotsProxy().isAuthorized()) return;
                updateLastScreenshotCapturedAt();
                if (bitmap != null) {
                    Parent lastParent = getLastParent();
                    InstabugInternalTrackingDelegate instance = InstabugInternalTrackingDelegate.getInstance();
                    if (instance != null && instance.getTargetActivity() != null && lastParent != null) {
                        String screenOrientation = getScreenOrientation(instance.getTargetActivity());
                        boolean isFlagSecure = ActivitySecureFlagDetector.isFlagSecure(instance.getTargetActivity());
                        lastParent.setScreenSecured(isFlagSecure);
                        saveBitmap(screenOrientation, bitmap, lastParent);
                        saveSessionReplayScreenshot(screenOrientation, bitmap, lastParent, screenName, null);
                    }
                } else {
                    if (getLastParent() != null) {
                        captureScreenshot(getLastParent(), screenName);
                        getLastParent().setHasOnResumeStep(true);
                    }
                }
            } catch (Exception e) {
                InstabugCore.reportError(e, "Error while adding VUS");
            }
        });
    }


    private void signalScreenshotCapturingStarted() {
        ReproStepsScreenshotEventBus.INSTANCE.post(ReproStepsScreenshotEventBus.SCREENSHOT_CAPTURING_STARTED);
    }

    private void signalScreenshotCapturingFinished() {
        ReproStepsScreenshotEventBus.INSTANCE.post(ReproStepsScreenshotEventBus.SCREENSHOT_CAPTURING_FINISHED);
    }

    @Override
    public void addVisualUserStep(@NotNull String stepType, @Nullable String screenName, @NotNull IBGUINode finalTarget, @NotNull Future<TouchedView> touchedViewFuture) {
        executor.execute(() -> {
            try {
                if (isPlatformNotValid()) return;
                TouchedView touchedView = touchedViewFuture != null ? touchedViewFuture.get() : null;
                if (touchedView == null) return;

                if (finalTarget.isTextField() && finalTarget.isFocusable()) {
                    return;
                }

                addStep(touchedView.getParent(),
                        stepType,
                        screenName,
                        touchedView.getProminentLabel(),
                        touchedView.getIconName()
                );
            } catch (Throwable throwable) {
                IBGDiagnostics.reportNonFatalAndLog(throwable, "Something Went Wrong While Adding VUS ", Constants.LOG_TAG);
            }

        });
    }

    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    @SuppressWarnings("java:S3776")
    @Override
    public void addVisualUserStep(@NonNull @StepType final String stepType,
                                  @Nullable final String screenName,
                                  @Nullable final String view,
                                  @Nullable final String icon) {

        signalScreenshotCapturingStarted();
        executor.execute(() -> {
            try {
                Parent parent = visualUserSteps.getLastParent();
                Parent updatedParent = null;

                if (screenName != null && screenName.equals("SupportRequestManagerFragment")) {
                    return;
                }

                // Avoid adding new steps coming from the app activities while the user is taking
                // an extra screenshot or recoding a video
                if (InstabugCore.isForegroundBusy() || shouldSkipNativeAutoCapturing()) {
                    signalScreenshotCapturingFinished();
                    return;
                }

                trim();

                switch (stepType) {
                    case ACTIVITY_RESUMED:
                    case FRAGMENT_RESUMED:
                    case DIALOG_FRAGMENT_RESUMED:
                        if (SettingsManager.getInstance().getCurrentPlatform() == Platform.FLUTTER) {
                            signalScreenshotCapturingFinished();
                            return;
                        }
                        if (parent != null && parent.getType() != null) {
                            if (parent.getScreenName() != null) {
                                if (parent.getScreenName().equals(screenName) && parent.getType().equals(stepType)) {
                                    InstabugInternalTrackingDelegate instance = InstabugInternalTrackingDelegate.getInstance();
                                    if (instance != null
                                            && instance.getTargetActivity() != null
                                            && parent.getScreenshot() != null
                                            && parent.getScreenshot().getId() != null) {
                                        String screenOrientation = getScreenOrientation(instance.getTargetActivity());
                                        saveSessionReplayScreenshot(screenOrientation, parent.getScreenshot().getId(), parent, view);
                                    }

                                    signalScreenshotCapturingFinished();
                                    return;
                                }
                            }

                            if (stepType.equals(ACTIVITY_RESUMED) && parent.getType().equals(FRAGMENT_RESUMED)
                                    && hasEmptySteps(parent)) {
                                signalScreenshotCapturingFinished();
                                return;
                            }
                            if (shouldIgnoreActivityStep(stepType, parent)) return;
                        }
                    case COMPOSE_STARTED:
                    case COMPOSE_RESUMED:

                        // A DialogFragment is a fragment that displays a dialog window, floating on top of
                        // its activity's window. Since they run in different Windows, touch events are
                        // never registered (we listen for touch events from the activity window).
                        // Therefor, hasEmptySteps(..) will always evaluate to true leading to ignoring
                        // important screenshots. This is a quick fix until we support multi window
                        // FIXME support multi window
                        if (parent != null && stepType.equals(DIALOG_FRAGMENT_RESUMED)) {
                            if (SystemClock.elapsedRealtime() - lastDialogFragmentDisplayedAt < 500
                                    || parent.isCapturingScreenShot()) {
                                parent.setScreenName(screenName);
                                signalScreenshotCapturingFinished();
                                return;
                            }
                            lastDialogFragmentDisplayedAt = SystemClock.elapsedRealtime();
                        } else if (parent != null && hasEmptySteps(parent)
                                && parent.getType() != null && !parent.getType().equals(FRAGMENT_RESUMED)
                                && !parent.getType().equals(COMPOSE_STARTED)
                                && !parent.getType().equals(COMPOSE_RESUMED)
                                && !parent.getType().equals(ACTIVITY_RESUMED)) {
                            parent.setScreenName(screenName);
                            if (getLastParent() != null) {
                                captureScreenshot(getLastParent(), view);
                            }
                            signalScreenshotCapturingFinished();
                            return;
                        }

                        if (stepType.equals(COMPOSE_STARTED) || stepType.equals(COMPOSE_RESUMED)) {
                            handleComposeEvent(stepType, parent, screenName, view);
                            return;
                        }

                        if ((parent == null || parent.hasOnResumeStep()) && platform != Platform.RN) {
                            addParent(screenName, stepType);
                            updatedParent = getCurrentParent();
                        }

                        if (updatedParent != null) {
                            updatedParent.setHasOnResumeStep(true);
                        }

                        // capture screenshot and add it to the last parent.
                        if (updatedParent != null && updatedParent.getScreenshot() == null) {
                            captureScreenshot(updatedParent, view);
                        }
                        break;
                    case TAB_SELECT:
                        if (screenName != null && !screenName.equals(lastScreenName)) {
                            addParent(screenName, stepType);
                            updatedParent = getCurrentParent();
                            if (updatedParent != null) {
                                captureScreenshot(updatedParent, view);
                            }
                        }
                        break;
                    case COMPOSE_STOPPED:
                        handleComposeEvent(stepType, parent, screenName, view);
                        break;
                    case ACTIVITY_DESTROYED:
                    case FRAGMENT_DETACHED:
                    case ACTIVITY_STOPPED:
                    case FRAGMENT_STOPPED:
                    case COMPOSE_DISPOSED:
                        if (parent != null) {
                            parent.setHasOnResumeStep(true);
                            VisualUserStep lastStep = parent.getLastStep();
                            if (lastStep != null) {
                                String stepType1 = lastStep.getStepType();
                                if (stepType1 != null && stepType1.equals(START_EDITING)) {
                                    logKeyboardEvent(parent, false);
                                }
                            }
                        }

                    case FRAGMENT_VISIBILITY_CHANGED:
                    case UNKNOWN:
                    case APPLICATION_CREATED:
                    case ACTIVITY_CREATED:
                    case ACTIVITY_STARTED:
                    case OPEN_DIALOG:
                    case FRAGMENT_ATTACHED:
                    case FRAGMENT_VIEW_CREATED:
                    case FRAGMENT_STARTED:
                        break;
                    case ACTIVITY_PAUSED:
                    case FRAGMENT_PAUSED:
                    case COMPOSE_PAUSED:
                    default:
                        addStep(parent, stepType, screenName, view, icon);
                        break;
                }

                lastScreenName = screenName;
                signalScreenshotCapturingFinished();

            } catch (Exception exception) {
                IBGDiagnostics.reportNonFatal(exception, "couldn't add visual user step");
            }
        });
    }

    private void handleComposeEvent(@NonNull @StepType String stepType, @Nullable Parent parent, @Nullable String screenName, @Nullable final String view) {
        handleComposeStarted(stepType, parent, screenName, view);
        handleComposeResumed(stepType, parent, view);
        handleComposeStopped(stepType);
    }

    private void handleComposeStarted(@NonNull @StepType String stepType, @Nullable Parent parent, @Nullable String screenName, @Nullable final String view) {
        if (!stepType.equals(COMPOSE_STARTED)) return;
        Parent updatedParent = null;
        if ((parent == null || parent.hasOnResumeStep()) && platform != Platform.RN) {
            addParent(screenName, stepType);
            updatedParent = getCurrentParent();
        }

        if (updatedParent != null) {
            updatedParent.setHasOnResumeStep(true);
        }
        InstabugInternalTrackingDelegate instance = InstabugInternalTrackingDelegate.getInstance();
        if (instance == null) return;
        Activity targetActivity = instance.getTargetActivity();
        if (targetActivity == null) return;
        SRScreenshotLog log = createSRScreenshotLog(getScreenOrientation(targetActivity), ActivitySecureFlagDetector.isFlagSecure(targetActivity), null, updatedParent, view);
        PoolProvider.postOrderedIOTask(SR_REPRO_EXEC_Q_ID, () -> {
            if (log != null)
                SessionReplayServiceLocator.getSessionReplayScreenshotStore().invoke(log);
        });
        if (updatedParent != null && log != null) {
            pendingComposeSteps.add(new Pair<>(updatedParent, log));
        }
    }

    private void handleComposeResumed(@NonNull @StepType String stepType, @Nullable Parent lastParent, @Nullable final String view) {
        Pair<Parent, SRScreenshotLog> pendingComposeStep = pendingComposeSteps.peekFirst();
        if (!stepType.equals(COMPOSE_RESUMED) || pendingComposeStep == null) return;
        Parent pendingComposeParent = pendingComposeStep.first;
        SRScreenshotLog pendingComposeLog = pendingComposeStep.second;
        if (lastParent != null && lastParent.getType() != null && !lastParent.getType().equals(COMPOSE_STARTED)) {
            visualUserSteps.removeParent(pendingComposeParent);
            visualUserSteps.addParent(pendingComposeParent);
        }
        pendingComposeParent.setType(COMPOSE_RESUMED);
        captureScreenshot(pendingComposeParent, view, 0, pendingComposeLog);
        pendingComposeSteps.removeFirst();
    }

    private void handleComposeStopped(@NonNull @StepType String stepType) {
        Pair<Parent, SRScreenshotLog> pendingComposeStep = pendingComposeSteps.peekLast();
        if (!stepType.equals(COMPOSE_STOPPED) || pendingComposeStep == null) return;
        pendingComposeSteps.removeLast();
    }

    private boolean shouldIgnoreActivityStep(String stepType, @NonNull Parent parent) {
        if (parent.getType() != null &&
                stepType.equals(ACTIVITY_RESUMED) &&
                (parent.getType().equals(COMPOSE_STARTED)
                        || parent.getType().equals(COMPOSE_RESUMED))
                && hasEmptySteps(parent)) {
            signalScreenshotCapturingFinished();
            return true;
        }
        return false;
    }

    private boolean shouldSkipNativeAutoCapturing() {
        return platform == Platform.UNITY
                || platform == Platform.RN;
    }

    private boolean isPlatformNotValid() {
        return platform == Platform.UNITY
                || platform == Platform.FLUTTER;
    }

    private boolean hasEmptySteps(Parent parent) {
        return parent.getSteps().isEmpty()
                || (parent.getSteps().size() == 1
                && parent.getSteps().getFirst().getStepType() != null
                && parent.getSteps().getFirst().getStepType().equals(APPLICATION_FOREGROUND));
    }

    @VisibleForTesting
    void addParent(@Nullable String screenName, @Nullable @StepType String type) {
        try {
            visualUserSteps.addParent(new Parent(String.valueOf(++screenId), screenName, type));
            if (pendingForeground != null && visualUserSteps.getLastParent() != null) {
                visualUserSteps.getLastParent().addStep(Builder(pendingForeground.getStepType())
                        .screenName(screenName)
                        .parentScreenId(visualUserSteps.getLastParent().getId())
                        .view("")
                        .isContainIcon(false)
                        .buttonIcon(null)
                        .build());
                pendingForeground = null;
            }
        } catch (Exception exception) {
            IBGDiagnostics.reportNonFatal(exception, "couldn't add Parent to visualUserSteps");
        }
    }

    private void addStep(@Nullable @StepType String stepType,
                         @Nullable String screenName, @Nullable String view, @Nullable String icon) {
        executor.execute(() -> addStep(visualUserSteps.getLastParent(), stepType, screenName, view, icon));
    }

    private void addStep(@Nullable Parent parent, @Nullable @StepType String stepType,
                         @Nullable String screenName, @Nullable String view, @Nullable String icon) {
        try {
            if (InstabugCore.isForegroundBusy()) return;
            // This fix for React Native
            if (parent == null) {
                if (shouldSkipNativeAutoCapturing()) {
                    return;
                }
                if (stepType != null && !stepType.equals(APPLICATION_BACKGROUND)) {
                    addParent(screenName, stepType);
                    parent = getCurrentParent();
                }
            }
            if (stepType != null) {
                if (stepType.equals(SCROLL) || stepType.equals(PINCH) || stepType.equals(SWIPE)) {
                    view = null;
                }
            }

            if (view == null) {
                view = "";
            }

            // If the first step in this parent is swipe and the last parent type is TAB_SELECT
            // Move this step to last parent. Learn more: https://instabug.atlassian.net/browse/IBGBUGCHAT-10174
            if (parent != null && stepType != null) {
                if ((stepType.equals(SWIPE) || stepType.equals(SCROLL))
                        && parent.getType() != null
                        && parent.getType().equals(TAB_SELECT)
                        && parent.getSteps().isEmpty()) {
                    Parent lastParent = getLastParent();
                    if (lastParent != null) {
                        stepType = SWIPE;
                        parent = lastParent;
                    }
                }
            }

            if (parent != null)
                visualUserSteps.addStep(parent, Builder(stepType)
                        .screenName(screenName)
                        .parentScreenId(parent.getId())
                        .view(view)
                        .isContainIcon(!TextUtils.isEmpty(icon))
                        .buttonIcon(icon)
                        .build());
        } catch (Exception exception) {
            IBGDiagnostics.reportNonFatal(exception, "couldn't add step to visualUsersSteps");
        }
    }

    @Override
    public void logKeyboardEvent(boolean isKeyboardOpen) {
        executor.execute(() -> {
            try {
                logKeyboardEvent(getCurrentParent(), isKeyboardOpen);
            } catch (Exception exception) {
                IBGDiagnostics.reportNonFatal(exception, "couldn't log keyboard event");
            }
        });

    }

    @SuppressLint("NULL_DEREFERENCE")
    private void logKeyboardEvent(@Nullable Parent parent, boolean isKeyboardOpen) {

        if (isKeyboardOpen
                && parent != null
                && parent.getLastStep() != null
                && parent.getLastStep().getStepType() != null
                && parent.getLastStep().getStepType().equals(START_EDITING)) {

            if (lastView != null) {
                String currentHint = getHint(lastView);
                String lastHint = parent.getLastStep().getView();
                if (lastHint != null && !lastHint.equals(currentHint)) {
                    addStep(END_EDITING,
                            parent.getLastStep().getScreenName(),
                            parent.getLastStep().getView(), null);
                }
            } else {
                return;
            }
        }

        addStep(parent, isKeyboardOpen ? START_EDITING : END_EDITING,
                lastScreenName, getHint(lastView), null);

    }

    @VisibleForTesting
    String getHint(@Nullable WeakReference<View> view) {
        if (view == null || view.get() == null)
            return DEFAULT_TEXT_FIELD;

        if (view.get() != null && view.get() instanceof EditText) {
            EditText lastEditText = (EditText) view.get();
            if (lastEditText != null) {
                try {
                    CharSequence hint = lastEditText.getHint();
                    CharSequence contentDescription = lastEditText.getContentDescription();
                    if (hint != null) {
                        if (!VisualUserStepsHelper.isPrivateView(lastEditText) && !TextUtils.isEmpty(hint.toString())) {
                            return lastEditText.getHint().toString();
                        }
                    } else if (contentDescription != null && !TextUtils.isEmpty(contentDescription.toString())) {
                        return lastEditText.getContentDescription().toString();
                    }
                } catch (NullPointerException | OutOfMemoryError throwable) {
                    return DEFAULT_TEXT_FIELD;
                }
            }
        }
        return DEFAULT_TEXT_FIELD;
    }

    @VisibleForTesting
    private void addForegroundEvent(@StepType String stepType) {
        Parent lastParent = getLastParent();
        if (lastParent != null && COMPOSE_STARTED.equals(lastParent.getType())) {
            lastParent.addStep(Builder(stepType)
                    .screenName(lastParent.getScreenName())
                    .parentScreenId(lastParent.getId())
                    .view("")
                    .isContainIcon(false)
                    .buttonIcon(null)
                    .build());
        } else {
            pendingForeground = Builder(stepType)
                    .screenName(null)
                    .parentScreenId(null)
                    .view("")
                    .isContainIcon(false)
                    .buttonIcon(null)
                    .build();
        }
    }

    @VisibleForTesting
    void captureScreenshot(final Parent lastParent, @Nullable String view, int delay, @Nullable SRScreenshotLog pendingLog) {
        if (lastParent.isCapturingScreenShot()) return;

        final Activity activity = InstabugInternalTrackingDelegate
                .getInstance()
                .getTargetActivity();

        signalScreenshotCapturingStarted();

        lastParent.setCapturingScreenShot(true);
        updateLastScreenshotCapturedAt();
        if (activity != null)
            captureScreenshot(lastParent, activity, view, delay, pendingLog);
    }

    @VisibleForTesting
    void captureScreenshot(final Parent lastParent, @Nullable String view) {
        captureScreenshot(lastParent, view, 500, null);
    }

    private void updateLastScreenshotCapturedAt() {
        lastScreenshotCapturedAt = TimeUtils.currentTimeMillis();
    }

    private void captureScreenshot(Parent lastParent, @NonNull Activity activity, @Nullable String view, int delay, @Nullable SRScreenshotLog pendingLog) {
        WeakReference<Activity> activityReference = new WeakReference<>(activity);
        SimpleCompletableFuture<Bitmap> screenshotFuture = new SimpleCompletableFuture<>();
        new Handler(Looper.getMainLooper()).postDelayed(() -> {
                    if (activityReference.get() != null) {
                        CoreServiceLocator.getReproScreenshotsProxy().capture(
                                ScreenshotRequest.createScreenshotRequest(
                                        new ScreenshotRequestArgs(
                                                VISUAL_USER_STEP_SCREENSHOT,
                                                activityReference.get(),
                                                createCapturingListener(lastParent, activityReference.get(), screenshotFuture))));
                    }
                },
                delay);
        saveSessionReplayScreenshot(getScreenOrientation(activity), screenshotFuture, lastParent, view, pendingLog);
    }

    @NonNull
    private ScreenshotCaptor.CapturingCallback createCapturingListener(Parent lastParent, @NonNull Activity activity, @NonNull SimpleCompletableFuture<Bitmap> screenshotFuture) {
        String orientation = getScreenOrientation(activity);
        boolean isFlagSecure = ActivitySecureFlagDetector.isFlagSecure(activity);
        lastParent.setScreenSecured(isFlagSecure);
        return new ScreenshotCaptor.CapturingCallback() {
            @Override
            public void onCapturingSuccess(@NonNull Bitmap bitmap) {
                ScreenshotsAnalyticsEventBus.INSTANCE.post(AnalyticsEvent.ScreenshotEvent.NewScreenshotCaptured.INSTANCE);
                lastParent.setCapturingScreenShot(false);
                saveBitmap(orientation, bitmap, lastParent);
                screenshotFuture.complete(bitmap);
            }

            @Override
            public void onCapturingFailure(@NonNull Throwable throwable) {
                if (throwable instanceof IBGLowMemroyWarning || throwable instanceof OutOfMemoryError) {
                    ScreenshotsAnalyticsEventBus.INSTANCE.post(new AnalyticsEvent.ErrorEvent(ScreenshotsErrorCodes.LOW_MEMORY, throwable));
                } else if (!(throwable instanceof ReproCapturingNotAuthorizedException)) {
                    ScreenshotsAnalyticsEventBus.INSTANCE.post(new AnalyticsEvent.ErrorEvent(ScreenshotsErrorCodes.UNKNOWN_EXCEPTION, throwable));
                }
                lastParent.setCapturingScreenShot(false);
                InstabugSDKLogger.d(Constants.LOG_TAG,
                        "capturing VisualUserStep failed error: " +
                                throwable.getMessage());
                signalScreenshotCapturingFinished();
                screenshotFuture.completeExceptionally(throwable);
            }
        };
    }

    @VisibleForTesting
    void saveBitmap(final String orientation, final Bitmap bitmap, final Parent lastParent) {
        if (InstabugFeaturesManager.getInstance().isFeatureAvailable(IBGFeature.REPRO_STEPS))
            saveVisualStepScreenshot(orientation, bitmap, lastParent);
    }

    private void saveVisualStepScreenshot(final String orientation, Bitmap bitmap, Parent lastParent) {
        PoolProvider.postIOTask(() -> {
            InstabugSDKLogger.d(Constants.LOG_TAG, "Saving bitmap for user step " + STEP_FILE_PREFIX + lastParent.getId());
            try {
                Uri screenshotUri = BitmapUtils.saveBitmapAsPNG(bitmap, SCREENSHOT_QUALITY_PERCENTAGE, CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory(),
                        STEP_FILE_PREFIX + lastParent.getId());
                Parent.Screenshot screenshot = new Parent.Screenshot(screenshotUri.getLastPathSegment());
                screenshot.setViewOrientation(orientation);
                lastParent.setScreenshot(screenshot);

                if (screenshotUri.getPath() != null) {
                    InstabugCore.encryptBeforeMarshmallow(screenshotUri.getPath());
                }
                signalScreenshotCapturingFinished();
            } catch (Throwable throwable) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "capturing VisualUserStep failed error: " + throwable.getMessage());
                signalScreenshotCapturingFinished();
                if (throwable instanceof OutOfMemoryError) {
                    ScreenshotsAnalyticsEventBus.INSTANCE.post(new AnalyticsEvent.ErrorEvent(ScreenshotsErrorCodes.LOW_MEMORY, throwable));
                } else if (throwable instanceof IOException) {
                    handleIOException((IOException) throwable, bitmap);
                } else {
                    ScreenshotsAnalyticsEventBus.INSTANCE.post(new AnalyticsEvent.ErrorEvent(ScreenshotsErrorCodes.UNKNOWN_EXCEPTION, throwable));
                }
            }
        });
    }

    private void handleIOException(IOException throwable, Bitmap bitmap) {
        boolean specificSpcex = false;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (throwable.getCause() instanceof ErrnoException) {
                int errno = 0;
                errno = ((ErrnoException) throwable.getCause()).errno;
                specificSpcex = errno == OsConstants.ENOSPC;
            }
        }
        boolean isNoStorage = specificSpcex || DeviceStateProvider.getFreeInternalStorage() <=
                BitmapUtils.getCompressedBitmapSize(bitmap, SCREENSHOT_QUALITY_PERCENTAGE);

        if (isNoStorage) {
            ScreenshotsAnalyticsEventBus.INSTANCE.post(new AnalyticsEvent.ErrorEvent(ScreenshotsErrorCodes.NO_STORAGE, throwable));
        } else {
            ScreenshotsAnalyticsEventBus.INSTANCE.post(new AnalyticsEvent.ErrorEvent(ScreenshotsErrorCodes.UNKNOWN_EXCEPTION, throwable));
        }
    }

    @WorkerThread
    private void saveSessionReplayScreenshot(final String orientation, @Nullable Bitmap bitmap, Parent lastParent, @Nullable String view, @Nullable SRScreenshotLog pendingLog) {
        SRScreenshotStore screenshotsStore = SessionReplayServiceLocator.getSessionReplayScreenshotStore();
        if (pendingLog != null) {
            SRScreenshotLog pendingCopy = new SRScreenshotLog.Builder()
                    .fillFromInstance(pendingLog)
                    .setMetadataStored(true)
                    .setBitmap(bitmap)
                    .build();
            if (pendingCopy != null) screenshotsStore.invoke(pendingCopy);
        } else {
            SRScreenshotLog srScreenshotLog = createSRScreenshotLog(orientation, null, bitmap, lastParent, view);
            if (srScreenshotLog != null)
                screenshotsStore.invoke(srScreenshotLog);
        }
        resetLastScreenshotCapturedAt();
    }

    private void saveSessionReplayScreenshot(final String orientation, @NonNull SimpleCompletableFuture<Bitmap> bitmapFuture, Parent lastParent, @Nullable String view, @Nullable SRScreenshotLog pendingLog) {
        PoolProvider.postOrderedIOTask(SR_REPRO_EXEC_Q_ID, () -> {
            Bitmap screenshotBitmap = null;
            try {
                screenshotBitmap = bitmapFuture.get();
            } catch (Throwable t) {
                // Swallow
            }
            saveSessionReplayScreenshot(orientation, screenshotBitmap, lastParent, view, pendingLog);
        });
    }

    private void saveSessionReplayScreenshot(final String orientation, String bitmapFileName, Parent lastParent, @Nullable String view) {
        PoolProvider.postOrderedIOTask(SR_REPRO_EXEC_Q_ID, () -> {
            Bitmap bitmap = BitmapFactory.decodeFile(CoreServiceLocator.getReproScreenshotsCacheDir().getCurrentSpanDirectory() + "/" + bitmapFileName);
            saveSessionReplayScreenshot(orientation, bitmap, lastParent, view, null);
        });
    }

    private void resetLastScreenshotCapturedAt() {
        lastScreenshotCapturedAt = 0;
    }

    @Nullable
    private SRScreenshotLog createSRScreenshotLog(final String orientation, @Nullable Boolean isScreenSecured, @Nullable Bitmap bitmap, @Nullable Parent lastParent, @Nullable String view) {
        long timestamp = getTimestamp(lastParent);
        boolean isScreenshotSecured = false;
        boolean isManuallyTriggered = false;
        if (lastParent != null) {
            isScreenshotSecured = lastParent.isScreenSecured();
            isManuallyTriggered = lastParent.isManuallyInvoked();
        }
        if (isScreenSecured != null) {
            isScreenshotSecured = isScreenSecured;
        }
        return new SRScreenshotLog.Builder()
                .setBitmap(bitmap)
                .setScreenView(view)
                .setTimestamp(timestamp)
                .setViewOrientation(orientation)
                .setScreenSecure(isScreenshotSecured)
                .setMetadataStored(false)
                .setManuallyInvoked(isManuallyTriggered)
                .build();
    }

    private long getTimestamp(@Nullable Parent lastParent) {
        if (lastScreenshotCapturedAt != 0)
            return lastScreenshotCapturedAt;
        if (lastParent != null && lastParent.getLastStep() != null)
            return lastParent.getLastStep().getDate();

        return TimeUtils.currentTimeMillis();
    }

    @NonNull
    private static String getScreenOrientation(@Nullable Activity activity) {
        if (activity == null) return VisualUserStep.ViewOrientation.PORTRAIT;

        int orientation = activity.getResources().getConfiguration().orientation;

        return (orientation == Configuration.ORIENTATION_LANDSCAPE) ?
                VisualUserStep.ViewOrientation.LANDSCAPE :
                VisualUserStep.ViewOrientation.PORTRAIT;
    }

    @Override
    @NonNull
    public ArrayList<VisualUserStep> fetch() {
        ArrayList<VisualUserStep> steps = new ArrayList<>();
        for (Parent parent : visualUserSteps.getParents()) {

            Builder builder = Builder(null).screenName(parent.getScreenName())
                    .parentScreenId(null)
                    .screenId(parent.getId())
                    .setManuallyInvoked(parent.isManuallyInvoked())
                    .setScreenSecure(parent.isScreenSecured());


            if (parent.getScreenshot() != null) {
                builder.screenshotId(parent.getScreenshot().getId())
                        .viewOrientation(parent.getScreenshot().getViewOrientation());
            }

            steps.add(builder.build());

            steps.addAll(parent.getSteps());
        }
        return steps;
    }

    @WorkerThread
    private void trim() {
        try {
            trimScreenshots();
            removeIrrelevantSteps();
            trimSteps();
        } catch (Exception e) {
            InstabugCore.reportError(e, "Error while trimming reprosteps");
        }
    }

    @Override
    public void clean() {
        visualUserSteps.cleanDirectoryAsync();
        visualUserSteps.removeAllSteps();
    }

    @Override
    public void reset() {
        screenId = 0;
    }

    private void trimScreenshots() {
        try {
            if (visualUserSteps.getParentsCount() > getScreenshotsLimit()) {
                int countToRemove = visualUserSteps.getParentsCount() - getScreenshotsLimit();
                visualUserSteps.removeFirstParents(countToRemove);
            }
        } catch (Exception e) {
            InstabugCore.reportError(e, "Error while trimming screenshots");
        }
    }

    private void removeIrrelevantSteps() {
        // These steps were added to handle corner cases. We should reset them
        for (Parent parent : visualUserSteps.getParents()) {
            List<VisualUserStep> stepsToBeRemoved = new ArrayList<>();
            for (VisualUserStep step : parent.getSteps()) {
                if (step.getStepType() != null) {
                    if (step.getStepType().equals(ACTIVITY_PAUSED)
                            || step.getStepType().equals(FRAGMENT_PAUSED)
                            || step.getStepType().equals(DIALOG_FRAGMENT_RESUMED)) {
                        stepsToBeRemoved.add(step);
                    }
                }
            }
            visualUserSteps.removeIrrelevantSteps(parent, stepsToBeRemoved);
        }
    }

    private void trimSteps() {
        try {
            if (visualUserSteps.getStepsCount() > getEventsLimit() + VISUAL_USER_STEPS_EVENTS_TOLERANCE_RATE) {
                while (visualUserSteps.getStepsCount() > getEventsLimit()) {
                    visualUserSteps.removeFirstStep();
                }
            }
        } catch (Exception e) {
            InstabugCore.reportError(e, "Error while triming steps");
        }
    }

    @Override
    public void logForegroundStep() {
        executor.execute(() -> {
            if (!shouldCaptureNextAppForeground) return;
            addForegroundEvent(APPLICATION_FOREGROUND);
            shouldCaptureNextAppForeground = false;
        });
    }

    @Override
    public void logBackgroundStep() {
        Parent lastParent = getLastParent();
        /*Skip adding duplicate background step due to events from sessions v2 and v3 event
            this check should be removed after removing sessions v2
        */
        if (lastParent != null && lastParent.getLastStep() != null && APPLICATION_BACKGROUND.equals(lastParent.getLastStep().getStepType())) {
            return;
        }
        final String nullString = null;
        addVisualUserStep(APPLICATION_BACKGROUND, nullString, nullString, nullString);
        shouldCaptureNextAppForeground = true;
    }

    @Override
    public void logInstabugEnabledStep() {
        // Todo: consider this event as a ACTIVITY_RESUMED/FRAGMENT_RESUMED and capture a screenshot,
        //  actual behavior to be defined by product
        Object lastSeenView = InstabugCore.getLastSeenView();
        if (lastSeenView != null) {
            addVisualUserStep(lastSeenView instanceof Fragment ? FRAGMENT_RESUMED : ACTIVITY_RESUMED,
                    lastSeenView.getClass().getSimpleName(),
                    lastSeenView.getClass().getName(), null);
        }
    }

    @Override
    public void removeScreenshotId(String screenshotUri) {
        for (Parent parent : visualUserSteps.getParents()) {
            if (parent.getScreenshot() != null
                    && parent.getScreenshot().getId() != null
                    && parent.getScreenshot().getId().equals(screenshotUri)) {
                parent.getScreenshot().setId(null);
                break;
            }
        }
    }

    @Override
    public void logFocusChange(@Nullable View oldFocus, @Nullable View newFocus) {
        if (oldFocus != null) {
            addStep(END_EDITING, lastScreenName,
                    getHint(new WeakReference<>(oldFocus)), null);
        }

        if (newFocus != null) {
            addStep(START_EDITING, lastScreenName,
                    getHint(new WeakReference<>(newFocus)), null);
        } else {
            addStep(END_EDITING, lastScreenName,
                    getHint(oldFocus != null ? new WeakReference<>(oldFocus) : null), null);
        }
    }

    @Override
    public void removeLastTapStep() {
        try {
            visualUserSteps.removeLastTapStep();
        } catch (Exception e) {
            InstabugCore.reportError(e, "Error while removing last tap step");
        }
    }

    @Override
    public void duplicateCurrentParent() {
        executor.execute(() -> {
            try {
                if (!isScreenshotCapturingAuthorized()) {
                    return;
                }
                Parent currentParent = getLastParent();
                if (currentParent == null ||
                        APPLICATION_BACKGROUND.equals(getLastStepType(currentParent))) {
                    InstabugSDKLogger.d(
                            Constants.LOG_TAG,
                            "Manual Screenshot wasn't captured " +
                                    "as there is no app screen currently displayed"
                    );
                    return;
                }
                signalScreenshotCapturingStarted();
                String screenName = currentParent.getScreenName();
                addParent(screenName, currentParent.getType());
                Parent updatedCurrentParent = getLastParent();
                if (updatedCurrentParent != null) {
                    updatedCurrentParent.setManuallyInvoked();
                    captureScreenshot(updatedCurrentParent, screenName);
                }
                signalScreenshotCapturingFinished();
            } catch (Exception exception) {
                InstabugCore.reportError(exception, "Error while adding manual VUS");
            }
        });
    }

    private boolean isScreenshotCapturingAuthorized() {
        return getReproScreenshotsCapturingProxy().isAuthorized();
    }

    private int getScreenshotsLimit() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(VISUAL_USER_STEPS_SCREENSHOTS_LIMIT);
    }

    private int getEventsLimit() {
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(VISUAL_USER_STEPS_EVENTS_LIMIT);
    }

    private ReproScreenshotsCapturingProxy getReproScreenshotsCapturingProxy() {
        return CoreServiceLocator.getReproScreenshotsProxy();
    }

    @VisibleForTesting
    public static void resetVisualUserStepsInstance() {
        INSTANCE = null;
    }
}
