package com.instabug.library.view.pagerindicator;

import static android.view.ViewGroup.LayoutParams.MATCH_PARENT;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.os.Build;
import android.util.AttributeSet;
import android.view.Gravity;
import android.widget.RelativeLayout;

import androidx.annotation.Nullable;


import com.instabug.library.view.ViewUtils;

import java.util.ArrayList;

public final class DotIndicator extends RelativeLayout implements SelectionIndicator {
    private static final int DEFAULT_NUMBER_OF_DOTS = 1;
    private static final int DEFAULT_SELECTED_DOT_INDEX = 0;
    private static final int DEFAULT_UNSELECTED_DOT_DIAMETER_DP = 6;
    private static final int DEFAULT_SELECTED_DOT_DIAMETER_DP = 9;
    private static final int DEFAULT_UNSELECTED_DOT_COLOR = Color.WHITE;
    private static final int DEFAULT_SELECTED_DOT_COLOR = Color.WHITE;
    private static final int DEFAULT_SPACING_BETWEEN_DOTS_DP = 7;
    private static final int DEFAULT_DOT_TRANSITION_DURATION_MS = 200;
    private final ArrayList<Dot> dots = new ArrayList<>();
    private int numberOfDots;
    private int selectedDotIndex;
    private int unselectedDotDiameterPx;
    private int selectedDotDiameterPx;
    private int unselectedDotColor;
    private int selectedDotColor;
    private int spacingBetweenDotsPx;
    private int dotTransitionDuration;

    public DotIndicator(Context context) {
        this(context, null);
    }

    public DotIndicator(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public DotIndicator(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(attrs, defStyleAttr);
    }

    @SuppressLint({"ERADICATE_PARAMETER_NOT_NULLABLE", "CustomViewStyleable"})
    private void init(@Nullable AttributeSet attrs, int defStyleAttr) {
        // Use a TypedArray to process attrs
        final TypedArray attributes = getContext().obtainStyledAttributes(attrs, com.instabug.library.R.styleable
                .IBDotIndicator, defStyleAttr, 0);

        // Need to convert all default dimensions to px
        final int defaultSelectedDotDiameterPx =
                ViewUtils.convertDpToPx(getContext(), DEFAULT_SELECTED_DOT_DIAMETER_DP);
        final int defaultUnselectedDotDiameterPx = ViewUtils.convertDpToPx(getContext(),
                DEFAULT_UNSELECTED_DOT_DIAMETER_DP);
        final int defaultSpacingBetweenDotsPx =
                ViewUtils.convertDpToPx(getContext(), DEFAULT_SPACING_BETWEEN_DOTS_DP);

        // Assign provided attributes to member variables, or use the defaults if necessary
        numberOfDots = attributes
                .getInt(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerNumberOfDots, DEFAULT_NUMBER_OF_DOTS);
        selectedDotIndex = attributes
                .getInt(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerSelectedDotIndex,
                        DEFAULT_SELECTED_DOT_INDEX);
        unselectedDotDiameterPx = attributes
                .getDimensionPixelSize(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerUnselectedDotDiameter,
                        defaultUnselectedDotDiameterPx);
        selectedDotDiameterPx = attributes
                .getDimensionPixelSize(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerSelectedDotDiameter,
                        defaultSelectedDotDiameterPx);
        unselectedDotColor = attributes.getColor(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerUnselectedDotColor,
                DEFAULT_UNSELECTED_DOT_COLOR);
        selectedDotColor = attributes
                .getColor(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerSelectedDotColor,
                        DEFAULT_SELECTED_DOT_COLOR);
        spacingBetweenDotsPx = attributes
                .getDimensionPixelSize(com.instabug.library.R.styleable.IBDotIndicator_ibViewPagerSpacingBetweenDots,
                        defaultSpacingBetweenDotsPx);
        dotTransitionDuration = attributes
                .getDimensionPixelSize(com.instabug.library.R.styleable.IBDot_ibViewPagerTransitionDuration,
                        DEFAULT_DOT_TRANSITION_DURATION_MS);

        // Attributes are no longer required
        attributes.recycle();

        // Setup UI
        setLayoutParams(new LayoutParams(MATCH_PARENT, MATCH_PARENT));
        setGravity(Gravity.CENTER);
        reflectParametersInView();
    }

    private void reflectParametersInView() {
        // Reset the root View and the dot Collection so that the UI can be entirely recreated
        removeAllViews();
        dots.clear();

        // Create the dots incrementally from left to right
        for (int i = 0; i < numberOfDots; i++) {
            // Create a dot and set its properties
            final Dot dot = new Dot(getContext());
            dot.setInactiveDiameterPx(unselectedDotDiameterPx)
                    .setActiveDiameterPx(selectedDotDiameterPx)
                    .setActiveColor(selectedDotColor)
                    .setInactiveColor(unselectedDotColor)
                    .setTransitionDuration(dotTransitionDuration);


            // Make the dot active if necessary
            if (i == selectedDotIndex) {
                dot.setActive(false);
            } else {
                dot.setInactive(false);
            }

            // Create the positioning parameters
            final int maxDiameterDim = Math.max(selectedDotDiameterPx, unselectedDotDiameterPx);
            final int startMargin = i * (spacingBetweenDotsPx + unselectedDotDiameterPx);
            LayoutParams params = new LayoutParams(maxDiameterDim, maxDiameterDim);
            params.setMargins(startMargin, 0, 0, 0);

            // RTL layout support
            if (Build.VERSION.SDK_INT >= 17) {
                params.setMarginStart(startMargin);
            }

            // Apply the positioning parameters and add the dot to the UI
            dot.setLayoutParams(params);
            addView(dot);

            // Keep a record of the dot for later use
            dots.add(i, dot);
        }
    }

    public void redrawDots() {
        reflectParametersInView();
    }

    public void setUnselectedDotDiameterPx(final int unselectedDotDiameterPx) {
        this.unselectedDotDiameterPx = unselectedDotDiameterPx;
        reflectParametersInView();
    }

    public void setUnselectedDotDiameterDp(final int unselectedDotDiameterDp) {
        final int diameterPx = ViewUtils.convertDpToPx(getContext(), unselectedDotDiameterDp);
        setUnselectedDotDiameterPx(diameterPx);
    }

    public int getUnselectedDotDiameter() {
        return unselectedDotDiameterPx;
    }

    public void setSelectedDotDiameterPx(final int selectedDotDiameterPx) {
        this.selectedDotDiameterPx = selectedDotDiameterPx;
        reflectParametersInView();
    }

    public void setSelectedDotDiameterDp(final int selectedDotDiameterDp) {
        final int diameterPx = ViewUtils.convertDpToPx(getContext(), selectedDotDiameterDp);
        setSelectedDotDiameterPx(diameterPx);
    }

    public int getSelectedDotDiameter() {
        return selectedDotDiameterPx;
    }

    public int getUnselectedDotColor() {
        return unselectedDotColor;
    }

    public void setUnselectedDotColor(final int unselectedDotColor) {
        this.unselectedDotColor = unselectedDotColor;
        reflectParametersInView();
    }

    public int getSelectedDotColor() {
        return selectedDotColor;
    }

    public void setSelectedDotColor(final int selectedDotColor) {
        this.selectedDotColor = selectedDotColor;
        reflectParametersInView();
    }

    public void setSpacingBetweenDotsPx(final int spacingBetweenDotsPx) {
        this.spacingBetweenDotsPx = spacingBetweenDotsPx;
        reflectParametersInView();
    }

    public void setSpacingBetweenDotsDp(final int spacingBetweenDotsDp) {
        final int spacingPx = ViewUtils.convertDpToPx(getContext(), spacingBetweenDotsDp);
        setSpacingBetweenDotsPx(spacingPx);
    }

    public int getSpacingBetweenDots() {
        return spacingBetweenDotsPx;
    }

    @Override
    public void setSelectedItem(final int index, final boolean animate) {
        // If there are no dots, it doesn't make sense to perform an update
        if (!dots.isEmpty()) {
            try {
                // The previously selected dot may no longer exist if the number of dots has changed
                if (selectedDotIndex < dots.size()) {
                    dots.get(selectedDotIndex).setInactive(animate);
                }

                dots.get(index).setActive(animate);
            } catch (IndexOutOfBoundsException e) {
                // Catch and rethrow the exception to avoid showing the internal implementation
                throw new IndexOutOfBoundsException();
            }

            selectedDotIndex = index;
        }
    }

    @Override
    public int getSelectedItemIndex() {
        return selectedDotIndex;
    }

    @Override
    public int getNumberOfItems() {
        return numberOfDots;
    }

    @Override
    public void setNumberOfItems(final int numberOfItems) {
        numberOfDots = numberOfItems;
        reflectParametersInView();
    }

    @Override
    public int getTransitionDuration() {
        return dotTransitionDuration;
    }

    @Override
    public void setTransitionDuration(final int transitionDurationMs) {
        dotTransitionDuration = transitionDurationMs;
        reflectParametersInView();
    }

    @Override
    public void setVisibility(final boolean show) {
        setVisibility(show ? VISIBLE : INVISIBLE);
    }

    @Override
    public boolean isVisible() {
        return (getVisibility() == VISIBLE);
    }
}
