package com.instabug.library.user

import com.instabug.library.Constants
import com.instabug.library.Constants.LogPlaceHolders
import com.instabug.library.Feature
import com.instabug.library.IBGFeature
import com.instabug.library.Instabug
import com.instabug.library.InstabugFeaturesManager
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGCoreEventPublisher.post
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.plugin.PluginsManager
import com.instabug.library.internal.dataretention.DisposalPolicy
import com.instabug.library.internal.dataretention.RetentionContract
import com.instabug.library.internal.orchestrator.ActionsOrchestrator
import com.instabug.library.internal.orchestrator.ArchiveSessionCountAction
import com.instabug.library.internal.orchestrator.InvalidateCacheForIdentifyingUserAction
import com.instabug.library.internal.orchestrator.LinkPushTokenAction
import com.instabug.library.internal.orchestrator.LoginUserAction
import com.instabug.library.internal.orchestrator.LogoutUserAction
import com.instabug.library.internal.orchestrator.MigrateAnonymousUserEventsAction
import com.instabug.library.internal.orchestrator.MigrateUserAttributesAction
import com.instabug.library.internal.orchestrator.MigrateUserSessionsCountAction
import com.instabug.library.internal.orchestrator.PostUserLoggedInAction
import com.instabug.library.internal.orchestrator.UpdateLastSeenAction
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.internal.storage.cache.CacheManager
import com.instabug.library.internal.storage.cache.db.InstabugDBInsertionListener
import com.instabug.library.internal.storage.cache.db.InstabugDbContract
import com.instabug.library.internal.storage.cache.user.UserCacheManager
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.service.MigrateUUIDService
import com.instabug.library.session.SessionsLocalDataSource
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator.sessionCacheManger
import com.instabug.library.settings.SettingsManager
import com.instabug.library.user.handlepushtoken.PushNotificationRequestHelper
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.MD5Generator
import com.instabug.library.util.threading.PoolProvider
import io.reactivexport.observers.DisposableCompletableObserver
import org.json.JSONException
import java.util.UUID

/**
 * Created by Tarek360 on 1/11/17.
 */
object UserManager {
    @Volatile
    private var currentUUID: String? = null

    @JvmStatic
    var identifiedUserEmail: String
        get() {
            var email: String? = SettingsManager.getInstance().identifiedUserEmail
            if (email != null && email.isEmpty()) {
                email = SettingsManager.getInstance().enteredEmail
            }
            val emailLog =
                if (email.isNullOrEmpty()) LogPlaceHolders.EMPTY_EMAIL else LogPlaceHolders.NON_EMPTY_EMAIL
            InstabugSDKLogger.v(Constants.LOG_TAG, "getIdentifiedUserEmail: $emailLog")

            return email.orEmpty()
        }
        set(email) {
            SettingsManager.getInstance().identifiedUserEmail = email
            if ("" == email) {
                InstabugSDKLogger.d(
                    Constants.LOG_TAG,
                    "Email set to empty string, enabling user input of email"
                )
                return
            }
//            if (emailIValid(email)) {
//                InstabugSDKLogger.w(
//                    Constants.LOG_TAG,
//                    "Invalid email passed to setIdentifiedUserEmail, ignoring."
//                )
//            }
        }

    @JvmStatic
    var identifiedUsername: String?
        get() {
            var identifiedUsername = SettingsManager.getInstance().identifiedUsername
            if (identifiedUsername.isNullOrBlank()) {
                identifiedUsername = SettingsManager.getInstance().enteredUsername
            }
            val usernameLog =
                if (identifiedUsername.isNullOrBlank())
                    LogPlaceHolders.EMPTY_USERNAME
                else
                    LogPlaceHolders.NON_EMPTY_USERNAME
            InstabugSDKLogger.v(Constants.LOG_TAG, "getIdentifiedUsername: $usernameLog")
            return identifiedUsername
        }
        set(username) {
            val userNameLog =
                if (username.isNullOrBlank()) LogPlaceHolders.EMPTY_USERNAME else LogPlaceHolders.NON_EMPTY_USERNAME
            InstabugSDKLogger.v(Constants.LOG_TAG, "setIdentifiedUsername: $userNameLog")
            SettingsManager.getInstance().identifiedUsername = username
        }

    @JvmStatic
    var enteredUsername: String?
        get() = SettingsManager.getInstance().enteredUsername
        set(enteredUsername) {
            val usernameLog = if (enteredUsername.isNullOrBlank())
                LogPlaceHolders.EMPTY_USERNAME
            else
                LogPlaceHolders.NON_EMPTY_USERNAME
            InstabugSDKLogger.v(Constants.LOG_TAG, "setEnteredUsername: $usernameLog")
            SettingsManager.getInstance().enteredUsername = enteredUsername
        }

    @JvmStatic
    var enteredEmail: String?
        get() = SettingsManager.getInstance().enteredEmail
        set(enteredEmail) {
            val emailLog =
                if (enteredEmail.isNullOrBlank()) LogPlaceHolders.EMPTY_EMAIL else LogPlaceHolders.NON_EMPTY_EMAIL
            InstabugSDKLogger.v(Constants.LOG_TAG, "setEnteredEmail: $emailLog")
            SettingsManager.getInstance().setEnteredEmail(enteredEmail)
        }
    private val isUserHasActivity: Boolean
        get() {
            val isUserHasActivity = PluginsManager.getLastActivityTime() != 0L
            InstabugSDKLogger.v(Constants.LOG_TAG, "isUserHasActivity: $isUserHasActivity")
            return isUserHasActivity
        }


    @JvmStatic
    val uUIDBlocking: String?
        /**
         * returns UUID and performs inserting it to the database on the current thread
         * todo: find a cleaner fix to the thread racing error
         */
        get() {
            if (currentUUID != null) {
                return currentUUID
            }
            currentUUID = uUIDFromCache
            UserCacheManager.insertIfNotExists(currentUUID, sessionsCount)
            return currentUUID
        }


    @JvmStatic
    val sessionsCount: Int
        get() = SettingsManager.getInstance().sessionsCount

    @JvmStatic
    val isLoggedIn: Boolean
        get() = !SettingsManager.getInstance().isUserLoggedOut

    @Deprecated("")
    @JvmStatic
    val dataDisposalPolicy: DisposalPolicy
        get() = DisposalPolicy.create(
            InstabugDbContract.UserEntity.TABLE_NAME,
            InstabugDbContract.UserEntity.COLUMN_UUID,
            InstabugDbContract.UserEntity.COLUMN_LAST_SEEN,
            RetentionContract.USER_DATA
        )

    @JvmStatic
    val userName: String?
        get() = try {
            val enteredUsername = enteredUsername
            if (!enteredUsername.isNullOrBlank()) {
                enteredUsername
            } else {
                identifiedUsername
            }
        } catch (e: Exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error getting username$e")
            ""
        }

    @JvmStatic
    val userEmail: String
        get() {
            val enteredEmail = enteredEmail
            return if (!enteredEmail.isNullOrBlank()) {
                enteredEmail
            } else {
                identifiedUserEmail
            }
        }

    private val uUIDFromCache: String
        get() = synchronized(this) {
            var uuid = SettingsManager.getInstance().mD5Uuid
            if (uuid.isNullOrBlank()) {
                uuid = SettingsManager.getInstance().uuid
                if (uuid.isNullOrBlank()) {
                    uuid = UUID.randomUUID().toString()
                    if (SettingsManager.shouldLogExtraRequestData()) {
                        InstabugSDKLogger.v(Constants.LOG_TAG, "new randomly generated UUID: $uuid")
                    }
                    SettingsManager.getInstance().uuid = uuid
                }
            }
            uuid
        }

    @JvmStatic
    val emailForBugReport: String?
        get() {
            val enteredEmail = enteredEmail
            if (!enteredEmail.isNullOrBlank()) {
                return enteredEmail
            } else if (InstabugCore.getFeatureState(IBGFeature.CRASHES_CUSTOM_IDENTIFIED_EMAIL) == Feature.State.DISABLED) {
                return identifiedUserEmail
            }
            return null
        }

    /**
     * this method to identify user,
     * but first it checks if the given user name and email if not provided then it breaks
     * else if the given user and email already identified then it breaks,
     * else if the given user name and email are different than the already identified user
     * then it logout first then identify
     *
     * @param username .
     * @param email    .
     * @param userId  .
     * @param shouldValidateEmail used by old deprecated API ignore email validation
     */
    @JvmStatic
    @JvmOverloads
    fun identifyUser(
        username: String?,
        email: String?,
        userId: String? = null,
        shouldValidateEmail: Boolean = true
    ) {

        val hasValidId = CoreServiceLocator.userIdValidator(userId)
        val hasValidEmail = if (shouldValidateEmail)
            CoreServiceLocator.emailValidator(email)
        else
            !email.isNullOrBlank()
        if (!hasValidId && !hasValidEmail) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Empty email and Empty Id, Can't identify user")
            return
        }

        val trimmedEmail = email?.trim()
        val trimmedId = userId?.trim()

        if (isLoggedIn && isSameUser(trimmedEmail, trimmedId))
            return
        else if (isLoggedIn)
            logoutUser(false)

        if (hasValidEmail && trimmedEmail != null) identifiedUserEmail = trimmedEmail

        identifiedUsername = username
        val uuid = if (hasValidId) trimmedId else trimmedEmail?.let(::generateMD5UUID)
        if (uuid != null) {
            logInUser(uuid)
            currentUUID = uuid
        }
        if (InstabugCore.getFeatureState(IBGFeature.CRASHES_CUSTOM_IDENTIFIED_EMAIL) == Feature.State.DISABLED)
            enteredEmail = trimmedEmail
        migrateUUID()
    }

    private fun migrateUUID() {
        PoolProvider.getUserActionsExecutor().execute {
            InstabugSDKLogger.d(Constants.LOG_TAG, "migrate UUID")
            val md5UUID = SettingsManager.getInstance().mD5Uuid
            if (isUserHasActivity) {
                startMigrationRequest(md5UUID)
            } else {
                clearUserActivities()
                if (md5UUID == null) {
                    InstabugSDKLogger.v(Constants.LOG_TAG, "New UUID is null")
                }
            }
        }
    }

    private fun startMigrationRequest(md5UUID: String?) {
        SettingsManager.getInstance().setShouldMakeUUIDMigrationRequest(true)
        try {
            val oldUUID = SettingsManager.getInstance().uuid
            if (oldUUID == null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "old uuid is null")
                return
            }
            if (md5UUID == null) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "New UUID is null")
                return
            }
            migrateSessions(oldUUID, md5UUID)
            MigrateUUIDService.getInstance().migrateUUID(
                oldUUID,
                md5UUID,
                createRequestCallback(oldUUID, md5UUID)
            )
        } catch (e: JSONException) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Something went wrong while do UUID migration request", e
            )
        }

    }

    private fun migrateSessions(oldUUID: String, md5UUID: String) {
        migrateV2Sessions(oldUUID, md5UUID)
        migrateV3Sessions(oldUUID, md5UUID)
    }

    private fun migrateV3Sessions(oldUUID: String, md5UUID: String) {
        sessionCacheManger
            .migrateUUID(oldUUID, md5UUID)
    }

    private fun migrateV2Sessions(oldUUID: String, md5UUID: String) {
        SessionsLocalDataSource()
            .migrateUUID(oldUUID, md5UUID)
            .subscribe(object : DisposableCompletableObserver() {
                override fun onComplete() {
                    //NO-OP
                }

                override fun onError(e: Throwable) {
                    InstabugSDKLogger.e(
                        Constants.LOG_TAG,
                        "Error while updating UUID in db" + e.message
                    )
                }
            })
    }

    private fun createRequestCallback(
        oldUUID: String?,
        md5UUID: String?
    ) = object : Request.Callbacks<String?, Throwable?> {
        override fun onSucceeded(response: String?) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "old uuid $oldUUID")
            InstabugSDKLogger.v(Constants.LOG_TAG, "md5uuid $md5UUID")
            clearUserActivities()
            SettingsManager.getInstance()
                .setShouldMakeUUIDMigrationRequest(false)
        }

        override fun onFailed(error: Throwable?) {
            SettingsManager.getInstance()
                .setShouldMakeUUIDMigrationRequest(true)
        }

    }

    @JvmStatic
    fun clearUserActivities() {
        InstabugSDKLogger.v(Constants.LOG_TAG, "clearing User Activities")
        SettingsManager.getInstance().lastContactedAt = 0L
        CacheManager.getInstance().invalidateAllCachesForIdentifyingUsers()
    }

    /**
     * currentUUID shouldn't be null
     */

    @JvmStatic
    fun getUUID() = currentUUID ?: restoreUUIDFromCache()

    /**
     * In logout user's data is reset to the default, but first any related data will be archived to
     * user's table, those data includes: session count
     * and user's interaction on survey/announcements [this is done by sending logout event to survey plugin]
     * @param shouldLinkPushToken Invalidate the push token for the current user and Associate it to the generated UUID if true,
     * else just invalidate the push token for the current user
     */
    @JvmStatic
    fun logoutUser(shouldLinkPushToken: Boolean = true) {

        enteredEmail = ""
        enteredUsername = ""
        if (SettingsManager.getInstance().identifiedUserEmail.isNullOrBlank()
            && SettingsManager.getInstance().identifiedUsername.isNullOrBlank()
        ) {
            return
        }
        //Post USER_LOGGED_OUT  Event
        post(IBGSdkCoreEvent.User.LoggedOut)
        InstabugCore.setPushNotificationTokenSent(false)
        val pushNotificationRequestHelper = PushNotificationRequestHelper()
        pushNotificationRequestHelper.handleSendPushTokenRequests("INVALID_TOKEN", null)
        val uuid = getUUID()
        currentUUID = UUID.randomUUID().toString()
        val sessionsCount = sessionsCount
        ActionsOrchestrator.obtainOrchestrator(PoolProvider.getUserActionsExecutor())
            .addWorkerThreadAction(ArchiveSessionCountAction(uuid, sessionsCount))
            .addSameThreadAction(LogoutUserAction(currentUUID))
            .addSameThreadAction { if (shouldLinkPushToken) LinkPushTokenAction().run() }
            .addWorkerThreadAction(UpdateLastSeenAction(uuid, System.currentTimeMillis()))
            .orchestrate()
    }

    private fun isSameUser(email: String?, userId: String?): Boolean {
        val sameUserId = userId != null && userId == currentUUID
        val identifiedUserEmail: String? = SettingsManager.getInstance().identifiedUserEmail
        val sameUserEmail = email != null && email.equals(identifiedUserEmail, true)

        return sameUserId && sameUserEmail
    }

    private fun logInUser(md5UUID: String) {
        //Post USER_LOGGED_IN Event
        val orchestrator =
            ActionsOrchestrator.obtainOrchestrator(PoolProvider.getUserActionsExecutor())
                .addWorkerThreadAction(MigrateUserSessionsCountAction(md5UUID))
                .addWorkerThreadAction(InvalidateCacheForIdentifyingUserAction(md5UUID))
                .addWorkerThreadAction(MigrateUserAttributesAction(md5UUID))
                .addWorkerThreadAction(MigrateAnonymousUserEventsAction(md5UUID))
                .addWorkerThreadAction(LoginUserAction(md5UUID))
                .addWorkerThreadAction(PostUserLoggedInAction())
        if (!isUserHasActivity) orchestrator.addWorkerThreadAction(LinkPushTokenAction())
        orchestrator.orchestrate()
    }

    private fun generateMD5UUID(email: String): String? {
        return MD5Generator.generateMD5(email + SettingsManager.getInstance().appToken)
    }

    @JvmStatic
    fun getUUIDAsync(listener: InstabugDBInsertionListener<String?>?) {
        if (currentUUID != null) {
            listener?.onDataInserted(currentUUID)
            return
        }
        currentUUID = uUIDFromCache
        PoolProvider.postIOTask {
            if (currentUUID != null) {
                UserCacheManager.insertIfNotExists(currentUUID, sessionsCount)
                listener?.onDataInserted(currentUUID)
            }
        }
    }

    private fun restoreUUIDFromCache(): String =
        uUIDFromCache
            .also { cachedId -> currentUUID = cachedId }
            .also(::insertIfNotExistsAsync)

    private fun insertIfNotExistsAsync(id: String) {
        PoolProvider
            .getUserActionsExecutor()
            .execute { UserCacheManager.insertIfNotExists(id, sessionsCount) }
    }

    @JvmStatic
    fun prepareUserState() {
        val context = Instabug.getApplicationContext()
        if (context != null && InstabugFeaturesManager.getInstance()
                .getFeatureState(IBGFeature.INSTABUG) == Feature.State.ENABLED && SettingsManager.getInstance()
                .shouldMakeUUIDMigrationRequest()
        ) {
            migrateUUID()
        }
        restoreUUIDFromCache()
    }
}

