package com.instabug.library.tracking

import com.instabug.library.model.UserStep
import com.instabug.library.sessionreplay.SRLogReceiver
import com.instabug.library.tracking.InstabugTrackingStepsProvider.USER_STEPS_LIMIT
import com.instabug.library.util.LimitConstraintApplier
import com.instabug.library.util.extenstions.runOrLogError
import java.util.Collections

const val ERROR_WHILE_GETTING_USER_TRACKING_STEPS = "Error while getting user steps: "
const val ERROR_WHILE_REMOVING_STEP_FROM_USER_TRACKING_STEPS =
    "Error while removing step from user steps"
const val ERROR_WHILE_ADDING_STEP_TO_USER_TRACKING_STEPS =
    "Error while adding step to user steps"

/**
 * provider for the tracking user steps
 * hold the user steps in memory to be used in the {@link com.instabug.library.model}
 * */
interface UserStepsProvider {
    /**
     * @returns a list of last {@link USER_STEPS_LIMIT }(100) user step
     * */
    val userSteps: List<UserStep>
}

class CoreUserStepHandler(
    initialUserStepArray: MutableList<UserStep>? = null,
    private val limitsApplier: LimitConstraintApplier? = null
) : SRLogReceiver<UserStep>, UserStepsProvider {
    private val _userSteps: MutableList<UserStep> =
        initialUserStepArray?.let(Collections::synchronizedList)
            ?: Collections.synchronizedList(ArrayList(getLimit()))
    override val userSteps
        get() = runOrLogError(errorMessage = ERROR_WHILE_GETTING_USER_TRACKING_STEPS) {
            _userSteps.toList()
        }.getOrDefault(emptyList())

    override fun invoke(log: UserStep) {
        _userSteps.apply {
            validateUserStepsLimit()
            addUserStepLog(log)
        }
    }

    private fun MutableList<UserStep>.validateUserStepsLimit() =
        runOrLogError(errorMessage = ERROR_WHILE_REMOVING_STEP_FROM_USER_TRACKING_STEPS) {
            if (size >= getLimit()) _userSteps.removeAt(0)
        }

    private fun MutableList<UserStep>.addUserStepLog(log: UserStep) = runOrLogError(
        errorMessage = ERROR_WHILE_ADDING_STEP_TO_USER_TRACKING_STEPS
    ) {
        add(log)
    }

    private fun getLimit() = limitsApplier?.applyConstraints(USER_STEPS_LIMIT) ?: USER_STEPS_LIMIT
}
