package com.instabug.library.settings;

import static com.instabug.library.diagnostics.customtraces.settings.CustomTracesSettingsResolver.DEFAULT_MAX_COUNT;
import static com.instabug.library.experiments.constants.Constants.EXPERIMENTS_STORE_LIMIT_FALLBACK;
import static com.instabug.library.settings.SettingsManager.INSTABUG_SHARED_PREF_NAME;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.BuildConfig;
import com.instabug.library.Instabug;
import com.instabug.library.diagnostics.DiagnosticsCoreEventHandler;
import com.instabug.library.encryption.EncryptorVersions;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.internal.sharedpreferences.CorePrefPropertyKt;
import com.instabug.library.internal.utils.PreferencesUtils;
import com.instabug.library.model.FeaturesCache;
import com.instabug.library.percentagefeatures.PercentageFeature;

import org.json.JSONException;

/**
 * Contains SDK settings that should be maintained across application termination and restart
 *
 * @author mSobhy
 */
public class PersistableSettings {

    private static final String IB_UUID = "ib_uuid";
    private static final String IB_MD5_UUID = "ib_md5_uuid";
    private static final String IB_USER_DATA = "ib_user_data";
    private static final String IB_APP_TOKEN = "ib_app_token";
    private static final String IB_IS_FIRST_RUN = "ib_first_run";
    private static final String IB_IS_FIRST_RUN_AFTER_ENCRYPTOR_UPDATE = "ib_first_run_after_updating_encryptor";
    private static final String IB_ENCRYPTOR_VERSION = "ib_encryptor_version";
    private static final String IB_FIRST_RUN_AT = "ib_first_run_at";
    private static final String IB_IS_FIRST_DISMISS = "ib_first_dismiss";
    private static final String IB_LAST_CONTACTED_AT = "last_contacted_at";
    private static final String IB_PUSH_NOTIFICATIONS_VISIBILITY = "ib_pn";
    private static final String IB_IS_DEVICE_REGISTERED = "ib_device_registered";
    private static final String IB_LAST_MIGRATION_VERSION = "last_migration_version";
    private static final String IB_IS_USER_LOGGED_OUT = "ib_is_user_logged_out";
    private static final String IB_SHOULD_MAKE_UUID_MIGRATION_REQUEST =
            "ib_should_make_uuid_migration_request";
    private static final String IB_IS_SDK_VERSION_SET = "ib_is_sdk_version_set";
    private static final String IB_SDK_VERSION = "ib_sdk_version";
    private static final String OS_VERSION = "os_version";
    private static final String IB_SESSIONS_COUNT = "ib_sessions_count";
    private static final String ENTERED_EMAIL = "entered_email";
    private static final String ENTERED_NAME = "entered_name";
    private static final String IDENTIFIED_EMAIL = "identified_email";
    private static final String IDENTIFIED_NAME = "identified_name";
    private static final String IB_SESSION_STATUS = "session_status";
    private static final String IB_ONBOARDING_SHOULD_SHOW = "should_show_onboarding";
    private static final String IB_APP_LAST_SEEN = "last_seen_timestamp";
    private static final String IB_FEATURES_TTL = "features_ttl";
    private static final String IB_FEATURES_HASH = "features_hash";
    private static final String IB_LOGGING_SETTINGS = "ib_logging_settings";
    private static final String IBC_IS_PUSH_NOTIFICATION_TOKEN_SENT = "ibc_is_push_notification_token_sent";
    private static final String IBC_PUSH_NOTIFICATION_TOKEN = "ibc_push_notification_token";
    private static final String IB_FEATURES_CACHE_SETTINGS = "ib_features_cache";
    private static final String IB_SESSIONS_SYNC_CONFIG = "ib_sessions_sync_configurations";
    private static final String IB_FIRST_SESSION = "ib_is_first_session";
    private static final String IB_VERSION_CODE = "ib_version_code";
    private static final String IB_IS_USERS_PAGE_ENABLED = "ib_is_users_page_enabled";
    private static final String APP_VERSION_FIRST_SEEN = "instabug_app_version_first_seen";
    private static final String LAST_APP_VERSION = "instabug_last_app_version";
    private static final String IB_NON_FATALS_SETTINGS = "ib_non_fatals_settings";
    private static final String IB_NON_FATALS_LAST_SYNC = "ib_non_fatals_last_sync";
    private static final String IB_DIAGNOSTICS_SYNC_INTERVAL = "ib_diagnostics_sync_interval";
    private static final String PERCENTAGE_SUFFIX = "_percentage";
    private static final String IB_LAST_FOREGROUND_TIME = "ib_last_foreground_time";
    private static final String IB_SESSION_STITCHING_TIMEOUT = "ib_session_stitching_timeout";
    private static final String IB_DEQUEUE_THRESHOLD = "ib_dequeue_threshold";
    private static final String IB_COMPLETION_THRESHOLD = "ib_completion_threshold";
    private static final String IB_LAST_REPORT_TIME = "ib_last_report_time";
    private static final String IB_FH_SENSITIVITY = "ib_fatal_hangs_sensitivity";
    private static final String IB_EXPERIMENTS_STORE_LIMIT = "ib_experiments_store_limit";
    private static final String IB_CUSTOM_TRACES_COUNT = "ib_custom_traces_count";


    private final SharedPreferences sharedPreferences;
    private SharedPreferences.Editor sharedPreferencesEditor;
    private static PersistableSettings persistableSettings;

    @SuppressLint("CommitPrefEdits")
    private PersistableSettings() {
        sharedPreferences = CorePrefPropertyKt.getCorePreferences();
        if (sharedPreferences != null) {
            sharedPreferencesEditor = sharedPreferences.edit();
        }
    }

    /**
     * Returns the current singleton instance of this class.
     *
     * @return a {@code PersistableSettings} instance
     */
    @Nullable
    public synchronized static PersistableSettings getInstance() {
        Context context = Instabug.getApplicationContext();
        if (persistableSettings == null && context != null)
            init();
        return persistableSettings;
    }

    // TODO: Use Proguard mapping and Reflection to call this method after obfuscation
    @VisibleForTesting
    public static void init() {
        persistableSettings = new PersistableSettings();
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    public static void tearDown() {
        persistableSettings = null;
    }

    public SharedPreferences getSharedPreferences() {
        return sharedPreferences;
    }

    @NonNull
    public String getIdentifiedUsername() {
        String identifiedUsername = sharedPreferences.getString(IDENTIFIED_NAME, "");
        return identifiedUsername != null ? identifiedUsername : "";
    }

    public void setIdentifiedUsername(@Nullable String userName) {
        if (sharedPreferencesEditor == null) return;
        if (userName == null) {
            sharedPreferencesEditor.remove(IDENTIFIED_NAME);
        } else {
            sharedPreferencesEditor.putString(IDENTIFIED_NAME, userName);
        }
        sharedPreferencesEditor.apply();
    }

    @NonNull
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String getIdentifiedUserEmail() {
        if (sharedPreferences == null) return "";
        return sharedPreferences.getString(IDENTIFIED_EMAIL, "");
    }
    public void setIdentifiedUserEmail(String identifiedUserEmail) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(IDENTIFIED_EMAIL, identifiedUserEmail);
        sharedPreferencesEditor.apply();
    }

    public boolean isDeviceRegistered() {
        if (sharedPreferences == null) return false;
        return sharedPreferences.getBoolean(IB_IS_DEVICE_REGISTERED, false);
    }

    public void setIsDeviceRegistered(boolean isDeviceRegistered) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_IS_DEVICE_REGISTERED, isDeviceRegistered).apply();
    }

    public boolean isFirstRun() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_IS_FIRST_RUN, true);
    }

    public void setIsFirstRun(boolean isFirstRun) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_IS_FIRST_RUN, isFirstRun).apply();
        sharedPreferences.edit().putLong(IB_FIRST_RUN_AT, System.currentTimeMillis()).apply();
    }

    long getFirstRunAt() {
        if (sharedPreferences == null) return 0L;
        return sharedPreferences.getLong(IB_FIRST_RUN_AT, 0L);
    }

    void setFirstRunAt(long time) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putLong(IB_FIRST_RUN_AT, time).apply();
    }

    public long getLastContactedAt() {
        if (sharedPreferences == null) return 0L;
        return sharedPreferences.getLong(IB_LAST_CONTACTED_AT, 0L);
    }

    public void setLastContactedAt(long lastContactedAt) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putLong(IB_LAST_CONTACTED_AT, lastContactedAt).apply();
    }

    public boolean isAppOnForeground() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_PUSH_NOTIFICATIONS_VISIBILITY, true);
    }

    public void setIsAppOnForeground(boolean isAppOnForeground) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_PUSH_NOTIFICATIONS_VISIBILITY, isAppOnForeground)
                .apply();
    }

    public int getLastMigrationVersion() {
        if (sharedPreferences == null) return 0;
        return sharedPreferences.getInt(IB_LAST_MIGRATION_VERSION, 0);
    }

    public void setLastMigrationVersion(int lastMigrationVersion) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putInt(IB_LAST_MIGRATION_VERSION, lastMigrationVersion).apply();
    }

    public boolean isFirstDismiss() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_IS_FIRST_DISMISS, true);
    }

    public void setIsFirstDismiss(boolean isFirstDismiss) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_IS_FIRST_DISMISS, isFirstDismiss).apply();
    }

    public void setEnteredUsername(String enteredUsername) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(ENTERED_NAME, enteredUsername);
        sharedPreferencesEditor.apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public String getEnteredUsername() {
        if (sharedPreferences == null) return "";
        return sharedPreferences.getString(ENTERED_NAME, "");
    }

    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String getUuid() {
        if (sharedPreferences == null) return null;
        return sharedPreferences.getString(IB_UUID, null);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void setUuid(@Nullable String uuid) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IB_UUID, uuid).apply();
    }

    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String getMD5Uuid() {
        if (sharedPreferences == null) return null;
        return sharedPreferences.getString(IB_MD5_UUID, null);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void setMD5Uuid(@Nullable String uuid) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IB_MD5_UUID, uuid).apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public String getUserData() {
        if (sharedPreferences == null) return "";
        return sharedPreferences.getString(IB_USER_DATA, "");
    }

    public void setUserData(String userData) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IB_USER_DATA, userData).apply();
    }

    public boolean isUserLoggedOut() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_IS_USER_LOGGED_OUT, true);
    }

    public void setUserLoggedOut(boolean isUserLoggedOut) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_IS_USER_LOGGED_OUT, isUserLoggedOut).apply();
    }

    public boolean shouldMakeUUIDMigrationRequest() {
        if (sharedPreferences == null) return false;
        return sharedPreferences.getBoolean(IB_SHOULD_MAKE_UUID_MIGRATION_REQUEST, false);
    }

    public void setShouldMakeUUIDMigrationRequest(boolean shouldMigrate) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_SHOULD_MAKE_UUID_MIGRATION_REQUEST, shouldMigrate)
                .apply();
    }

    public void setCurrentSDKVersion(String sdkVersion) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IB_SDK_VERSION, sdkVersion).apply();
        sharedPreferences.edit().putBoolean(IB_IS_SDK_VERSION_SET, true).apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public String getLastSDKVersion() {
        if (sharedPreferences == null) return BuildConfig.SDK_VERSION;
        return sharedPreferences.getString(IB_SDK_VERSION, BuildConfig.SDK_VERSION);
    }

    public boolean isSDKVersionSet() {
        if (sharedPreferences == null) return false;
        return sharedPreferences.getBoolean(IB_IS_SDK_VERSION_SET, false);
    }

    public void setAppToken(String appToken) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(IB_APP_TOKEN, appToken).apply();
    }

    @Nullable
    public String getAppToken() {
        if (sharedPreferences == null) return null;
        return sharedPreferences.getString(IB_APP_TOKEN, null);
    }

    public void setOsVersion(int osVersion) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putInt(OS_VERSION, osVersion).apply();
    }

    public int getOsVersion() {
        if (sharedPreferences == null) return -1;
        return sharedPreferences.getInt(OS_VERSION, -1);
    }

    public int getSessionsCount() {
        if (sharedPreferences == null) return 0;
        return sharedPreferences.getInt(IB_SESSIONS_COUNT, 0);
    }

    public void incrementSessionsCount() {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putInt(IB_SESSIONS_COUNT, getSessionsCount() + 1).apply();
    }

    public void setEnteredEmail(String enteredEmail) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(ENTERED_EMAIL, enteredEmail);
        sharedPreferencesEditor.apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public String getEnteredEmail() {
        if (sharedPreferences == null) return "";
        return sharedPreferences.getString(ENTERED_EMAIL, "");
    }

    public void setIsSessionEnabled(boolean enabled) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_SESSION_STATUS, enabled).apply();
    }

    public boolean isSessionEnabled() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_SESSION_STATUS, true);
    }

    public void setShouldAutoShowOnboarding(boolean shouldShowOnboarding) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IB_ONBOARDING_SHOULD_SHOW, shouldShowOnboarding).apply();
    }

    public boolean shouldAutoShowOnboarding() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_ONBOARDING_SHOULD_SHOW, true);
    }

    public void setLastSeenTimestamp(long currentTimeMillis) {
        if (sharedPreferences == null) return;
        SharedPreferences.Editor edit = sharedPreferences.edit();
        edit.putLong(IB_APP_LAST_SEEN, currentTimeMillis).apply();
    }

    public long getLastSeenTimestamp() {
        if (sharedPreferences == null) return System.currentTimeMillis();
        return sharedPreferences.getLong(IB_APP_LAST_SEEN, System.currentTimeMillis());
    }

    public void resetSessionCount() {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putInt(IB_SESSIONS_COUNT, 0).apply();
    }

    public void setSessionCount(int sessionCount) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putInt(IB_SESSIONS_COUNT, sessionCount).apply();
    }

    public long getFeaturesTTL() {
        if (sharedPreferences == null) return 0;
        return sharedPreferences.getLong(IB_FEATURES_TTL, 0);
    }

    public void setFeaturesTTL(long ttl) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putLong(IB_FEATURES_TTL, ttl).apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public String getFeaturesHash() {
        if (sharedPreferences == null) return "";
        return sharedPreferences.getString(IB_FEATURES_HASH, "");
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void setFeaturesHash(@Nullable String hash) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IB_FEATURES_HASH, hash).apply();
    }

    public void setLoggingFeatureSettings(@Nullable String loggingSettings) {
        if (sharedPreferences == null) return;
        if (loggingSettings == null) {
            sharedPreferences.edit().remove(IB_LOGGING_SETTINGS);
        }
        sharedPreferences.edit().putString(IB_LOGGING_SETTINGS, loggingSettings).apply();
    }

    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String getLoggingFeatureSettings() {
        if (sharedPreferences == null) return null;
        return sharedPreferences.getString(IB_LOGGING_SETTINGS, null);
    }

    public FeaturesCache getFeaturesCache() throws JSONException {
        FeaturesCache featuresCache = new FeaturesCache();
        String cacheSettings = null;
        if (sharedPreferences != null)
            cacheSettings = sharedPreferences.getString(IB_FEATURES_CACHE_SETTINGS, null);
        featuresCache.fromJson(cacheSettings);
        return featuresCache;
    }

    void setFeaturesCache(FeaturesCache cache) throws JSONException {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IB_FEATURES_CACHE_SETTINGS, cache.toJson()).apply();
    }

    void setPushNotificationToken(String token) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putString(IBC_PUSH_NOTIFICATION_TOKEN, token).apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    String getPushNotificationToken() {
        if (sharedPreferences == null) return "";
        return sharedPreferences.getString(IBC_PUSH_NOTIFICATION_TOKEN, "");
    }

    void setPushNotificationTokenSent(boolean isSent) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putBoolean(IBC_IS_PUSH_NOTIFICATION_TOKEN_SENT, isSent).apply();
    }

    boolean isPushNotificationTokenSent() {
        if (sharedPreferences == null) return false;
        return sharedPreferences.getBoolean(IBC_IS_PUSH_NOTIFICATION_TOKEN_SENT, false);
    }


    public boolean isFirstRunAfterEncryptorUpdate() {
        // Defaults to true if not set before
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_IS_FIRST_RUN_AFTER_ENCRYPTOR_UPDATE, true);
    }

    public void setFirstRunAfterEncryptorUpdate(boolean isFirstRun) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putBoolean(IB_IS_FIRST_RUN_AFTER_ENCRYPTOR_UPDATE, isFirstRun);
        sharedPreferencesEditor.apply();
    }

    public void setSessionsSyncConfigurations(String json) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(IB_SESSIONS_SYNC_CONFIG, json).apply();
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public String getSessionsSyncConfigurations() {
        if (sharedPreferences == null) return "{}";
        return sharedPreferences.getString(IB_SESSIONS_SYNC_CONFIG, "{}");
    }

    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static String getSessionsSyncConfigurations(@NonNull Context context) {
        // This method might get called before building instabug completes.
        // Therefore, we can't rely on Instabug.getApplicationContext()
        PreferencesUtils preferencesUtils = new PreferencesUtils(context, INSTABUG_SHARED_PREF_NAME);
        return preferencesUtils.getString(IB_SESSIONS_SYNC_CONFIG, "{}");
    }

    public boolean isFirstSession() {
        if (sharedPreferences == null) return true;
        return sharedPreferences.getBoolean(IB_FIRST_SESSION, true);
    }

    public void setIsFirstSession(boolean firstSession) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putBoolean(IB_FIRST_SESSION, firstSession).apply();
    }

    public void setVersionCode(int versionCode) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putInt(IB_VERSION_CODE, versionCode).apply();
    }

    public int getLastKnownVersionCode() {
        if (sharedPreferences != null && sharedPreferences.getInt(IB_VERSION_CODE, -1) == -1) {
            setVersionCode(InstabugDeviceProperties.getVersionCode());
        }
        if (sharedPreferences == null) return -1;
        return sharedPreferences.getInt(IB_VERSION_CODE, -1);
    }

    public void setUsersPageEnabled(boolean enabled) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putBoolean(IB_IS_USERS_PAGE_ENABLED, enabled);
        sharedPreferencesEditor.apply();
    }

    public boolean isUsersPageEnabled() {
        if (sharedPreferences == null) return false;
        return sharedPreferences.getBoolean(IB_IS_USERS_PAGE_ENABLED, false);
    }

    public boolean getBoolean(String key, boolean defaultValue) {
        if (sharedPreferences == null) return defaultValue;
        return sharedPreferences.getBoolean(key, defaultValue);
    }

    public void saveBoolean(String key, boolean value) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putBoolean(key, value);
        sharedPreferencesEditor.apply();
    }

    public void saveCustomTracesCount(int count) {
        if (sharedPreferencesEditor == null)
            return;
        sharedPreferencesEditor.putInt(IB_CUSTOM_TRACES_COUNT, count);
        sharedPreferencesEditor.apply();
    }

    @Nullable
    public int getCustomTracesCount() {
        if (sharedPreferences == null)
            return DEFAULT_MAX_COUNT;
        return sharedPreferences.getInt(IB_CUSTOM_TRACES_COUNT, DEFAULT_MAX_COUNT);
    }

    public void saveEncryptorVersion(int version) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putInt(IB_ENCRYPTOR_VERSION, version);
        sharedPreferencesEditor.apply();

    }

    public int getEncryptorVersion() {
        if (sharedPreferences == null) return EncryptorVersions.ENCRYPTOR_V1;
        return sharedPreferences.getInt(IB_ENCRYPTOR_VERSION, EncryptorVersions.ENCRYPTOR_V1);
    }

    public long getFirstSeen() {
        if (sharedPreferences == null) return -1;
        return sharedPreferences.getLong(APP_VERSION_FIRST_SEEN, -1);
    }

    public void setFirstSeen(long firstSeen) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putLong(APP_VERSION_FIRST_SEEN, firstSeen);
        sharedPreferencesEditor.apply();
    }

    @Nullable
    public String getLastAppVersion() {
        if (sharedPreferences == null) return null;
        return sharedPreferences.getString(LAST_APP_VERSION, null);
    }

    public void setLastAppVersion(@Nullable String lastAppVersion) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(LAST_APP_VERSION, lastAppVersion);
        sharedPreferencesEditor.apply();
    }

    @Nullable
    public String getNonFatalsFeatureSettings() {
        if (sharedPreferences == null) return null;
        return sharedPreferences.getString(IB_NON_FATALS_SETTINGS, null);
    }

    public void setNonFatalsFeatureSettings(String nonFatalsFeatureSettings) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putString(IB_NON_FATALS_SETTINGS, nonFatalsFeatureSettings).apply();
    }

    @Nullable
    public long getDiagnosticsLastSyncTime() {
        if (sharedPreferences == null) return 0L;
        return sharedPreferences.getLong(IB_NON_FATALS_LAST_SYNC, 0L);
    }

    public void setDiagnosticsLastSyncTime(long timeMillis) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putLong(IB_NON_FATALS_LAST_SYNC, timeMillis).apply();
    }

    public void setDiagnosticsSyncInterval(int syncInterval) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putInt(IB_DIAGNOSTICS_SYNC_INTERVAL, syncInterval);
        sharedPreferencesEditor.apply();
    }

    public int getDiagnosticsSyncInterval() {
        if (sharedPreferences == null) return DiagnosticsCoreEventHandler.DEFAULT_SYNC_INTERVAL;
        return sharedPreferences.getInt(IB_DIAGNOSTICS_SYNC_INTERVAL, DiagnosticsCoreEventHandler.DEFAULT_SYNC_INTERVAL);
    }

    public void savePercentageFeature(@Nullable String featureName, @Nullable PercentageFeature feature) {
        if (feature != null && featureName != null && sharedPreferencesEditor != null) {
            sharedPreferencesEditor.putString(featureName + PERCENTAGE_SUFFIX, feature.toJson());
            sharedPreferencesEditor.apply();
        }
    }

    public void clearPercentageFeature(@Nullable String featureName) {
        if (featureName != null) {
            sharedPreferencesEditor.remove(featureName + PERCENTAGE_SUFFIX);
            sharedPreferencesEditor.apply();
        }
    }

    public void clearValue(String featureName) {
        sharedPreferencesEditor.remove(featureName);
        sharedPreferencesEditor.apply();
    }

    public PercentageFeature getPercentageFeature(@Nullable String featureName) {
        PercentageFeature feature = new PercentageFeature();
        if (featureName != null) {
            String jsonValue = "";
            if (sharedPreferences != null) {
                jsonValue = sharedPreferences.getString(featureName + PERCENTAGE_SUFFIX, "");
            }
            if (jsonValue != null) {
                feature.fromJson(jsonValue);
            }
        }

        return feature;
    }

    public long getLastForegroundTime() {
        if (sharedPreferences == null) return -1L;
        return sharedPreferences.getLong(IB_LAST_FOREGROUND_TIME, -1L);
    }

    public void setLastForegroundTime(long timeMillis) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putLong(IB_LAST_FOREGROUND_TIME, timeMillis).commit();
    }

    public int getSessionStitchingTimeout(int defaultSessionTimeout) {
        if (sharedPreferences == null) return defaultSessionTimeout;
        return sharedPreferences.getInt(IB_SESSION_STITCHING_TIMEOUT, defaultSessionTimeout);
    }

    public void setSessionStitchingTimeout(int sessionStitchingTimeout) {
        if (sharedPreferences == null) return;
        sharedPreferences.edit().putInt(IB_SESSION_STITCHING_TIMEOUT, sessionStitchingTimeout).commit();
    }

    public void saveDequeueThreshold(long dequeueThreshold) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putLong(IB_DEQUEUE_THRESHOLD, dequeueThreshold).commit();
    }

    public void saveCompletionThreshold(long completionThreshold) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putLong(IB_COMPLETION_THRESHOLD, completionThreshold).commit();
    }

    public void saveLastReportTime(long completionThreshold) {
        sharedPreferencesEditor.putLong(IB_LAST_REPORT_TIME, completionThreshold).commit();
    }

    public long getLastReportTime() {
        return sharedPreferences.getLong(IB_LAST_REPORT_TIME, 0L);
    }

    public long getDequeueThreshold() {
        return sharedPreferences.getLong(IB_DEQUEUE_THRESHOLD, 0L);
    }

    public long getCompletionThreshold() {
        return sharedPreferences.getLong(IB_COMPLETION_THRESHOLD, 0L);
    }

    public void setExperimentsStoreLimit(int storeLimit) {
        if (sharedPreferencesEditor == null) return;
        sharedPreferencesEditor.putInt(IB_EXPERIMENTS_STORE_LIMIT, storeLimit).commit();
    }

    public int getExperimentsStoreLimit() {
        if (sharedPreferences == null) return EXPERIMENTS_STORE_LIMIT_FALLBACK;
        return sharedPreferences.getInt(IB_EXPERIMENTS_STORE_LIMIT, EXPERIMENTS_STORE_LIMIT_FALLBACK);
    }

    public void setFatalHangsSensitivity(long sensitivity) {
        sharedPreferencesEditor.putLong(IB_FH_SENSITIVITY, sensitivity).commit();
    }

    public long getFatalHangsSensitivity() {
        return sharedPreferences.getLong(IB_FH_SENSITIVITY, 2000);
    }

    public boolean containsKey(String key) {
        return sharedPreferences.contains(key);
    }
}