package com.instabug.library.sessionreplay.configurations

import com.instabug.library.IssueType
import com.instabug.library.ReproMode
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.sessionreplay.INSTABUG_LOG_LIMIT_DEFAULT
import com.instabug.library.sessionreplay.INSTABUG_LOG_LIMIT_KEY
import com.instabug.library.sessionreplay.NETWORK_LIMIT_DEFAULT
import com.instabug.library.sessionreplay.NETWORK_LOG_LIMIT_KEY
import com.instabug.library.sessionreplay.SR_CALLBACK_ENABLED_DEFAULT
import com.instabug.library.sessionreplay.SR_CALLBACK_KEY
import com.instabug.library.sessionreplay.SR_ENABLED_DEFAULT
import com.instabug.library.sessionreplay.SR_ENABLED_KEY
import com.instabug.library.sessionreplay.SR_INSTABUG_LOG_DEFAULT
import com.instabug.library.sessionreplay.SR_INSTABUG_LOG_KEY
import com.instabug.library.sessionreplay.SR_KEY
import com.instabug.library.sessionreplay.SR_MAX_LOGS_DEFAULT
import com.instabug.library.sessionreplay.SR_MAX_LOGS_KEY
import com.instabug.library.sessionreplay.SR_MAX_SCREENSHOTS_MP_PER_SESSION_KEY
import com.instabug.library.sessionreplay.SR_MAX_SCREENSHOTS_SIZE_PER_SESSION_DEFAULT
import com.instabug.library.sessionreplay.SR_MAX_SDK_MB_KEY
import com.instabug.library.sessionreplay.SR_MAX_SDK_SIZE_DEFAULT
import com.instabug.library.sessionreplay.SR_MAX_SESSION_MB_KEY
import com.instabug.library.sessionreplay.SR_MAX_SESSION_SIZE_DEFAULT
import com.instabug.library.sessionreplay.SR_MONITORING_AVAILABILITY_DEFAULT
import com.instabug.library.sessionreplay.SR_MONITORING_AVAILABILITY_KEY
import com.instabug.library.sessionreplay.SR_NETWORK_ENABLED_DEFAULT
import com.instabug.library.sessionreplay.SR_NETWORK_KEY
import com.instabug.library.sessionreplay.SR_SAMPLING_RATE_DEFAULT
import com.instabug.library.sessionreplay.SR_SAMPLING_RATE_KEY
import com.instabug.library.sessionreplay.SR_SCREENSHOTS_COMPRESSION_DEFAULT
import com.instabug.library.sessionreplay.SR_SCREENSHOTS_COMPRESSION_KEY
import com.instabug.library.sessionreplay.SR_SCREENSHOTS_ENABLED_DEFAULT
import com.instabug.library.sessionreplay.SR_SCREENSHOTS_KEY
import com.instabug.library.sessionreplay.SR_SESSION_LINK_PREFIX_DEFAULT
import com.instabug.library.sessionreplay.SR_SESSION_LINK_PREFIX_KEY
import com.instabug.library.sessionreplay.SR_USER_STEPS_ENABLED_DEFAULT
import com.instabug.library.sessionreplay.SR_USER_STEPS_KEY
import com.instabug.library.sessionreplay.SrPreferences
import com.instabug.library.sessionreplay.model.SRLogType
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.visualusersteps.ReproConfigurationsProvider
import com.instabug.library.visualusersteps.ReproProxyAuthID
import com.instabug.library.visualusersteps.ReproRuntimeConfigurationsHandler
import org.json.JSONObject

//region Error Messages
private const val SOMETHING_WENT_WRONG_WHILE_HANDLING_SESSION_REPLAY_CONFIGURATIONS_CHANGE =
    "Something Went Wrong While Handling Session Replay Configurations Change"

//endregion
interface SRConfigurationsProvider : ReproConfigurationsProvider {
    val srAvailable: Boolean
    var srEnabled: Boolean

    val networkLogsAvailable: Boolean
    var networkLogsRTEnabled: Boolean
    val networkLogsEnabled: Boolean
    val networkLogLimit: Int

    val ibgLogsAvailable: Boolean
    var ibgLogsRTEnabled: Boolean
    val ibgLogsEnabled: Boolean
    val ibgLogsLimit: Int

    val userStepsAvailable: Boolean
    var userStepsRTEnabled: Boolean
    val userStepsEnabled: Boolean

    val maxLogs: Int

    /**
     * sampling rate in seconds
     *
     * */
    val samplingRate: Int

    val screenshotsRTEnabled: Boolean
    val screenshotsCompressionQuality: Float

    val reproStepsRTEnabled: Boolean

    /*
    * max SDK size in MB
    * */
    val maxSDKSize: Float

    /*
    * max Session Size in MB
    * */
    val maxSessionSize: Float

    /*
    * max screenshots size per session in MB
    * */
    val maxScreenshotsSizePerSession: Float

    var isMonitoringAvailable: Boolean
    val sessionLinkPrefix: String?

    val isCallbackEnabled: Boolean
}

fun interface SRLimitationsProvider {
    operator fun get(@SRLogType logType: String): Int
}

interface SRConfigurationsHandler :
    SRConfigurationsProvider,
    SRLimitationsProvider,
    ReproRuntimeConfigurationsHandler, ColdLaunchProvider {
    companion object Factory {
        fun create(): SRConfigurationsHandler = SRConfigurationsHandlerImpl()
    }

    fun handleConfigurationsChange(newConfig: String)
}

class SRConfigurationsHandlerImpl : SRConfigurationsHandler {
    override var isColdAPPLaunch: Boolean = true
    override var srAvailable by corePref(SrPreferences.srAvailable)
        private set
    override var srEnabled: Boolean = true
        get() = field && srAvailable

    override var networkLogsAvailable: Boolean by corePref(SrPreferences.srNetworkEnabled)
        private set
    override var networkLogsRTEnabled: Boolean = true
    override val networkLogsEnabled: Boolean
        get() = networkLogsAvailable && networkLogsRTEnabled
    override var networkLogLimit by corePref(SrPreferences.networkLimit)
        private set
    override var ibgLogsAvailable by corePref(SrPreferences.srInstabugLogEnabled)
        private set
    override var ibgLogsRTEnabled: Boolean = true
    override val ibgLogsEnabled: Boolean
        get() = ibgLogsAvailable && ibgLogsRTEnabled
    override var ibgLogsLimit by corePref(SrPreferences.instabugLogsLimit)
        private set
    override var userStepsAvailable by corePref(SrPreferences.srUserStepsEnabled)
        private set
    override var userStepsRTEnabled: Boolean = true
    override val userStepsEnabled: Boolean
        get() = userStepsAvailable && userStepsRTEnabled
    override var isReproScreenshotsAvailable: Boolean by corePref(SrPreferences.screenshotsEnabled)
    override var screenshotsRTEnabled: Boolean = true
    override var screenshotsCompressionQuality: Float by corePref(SrPreferences.screenshotsCompression)
        private set
    override var reproStepsRTEnabled: Boolean = true
    override var maxSDKSize by corePref(SrPreferences.srMaxSDKSize)
        private set
    override var maxLogs by corePref(SrPreferences.srMaxLogs)
        private set
    override var samplingRate by corePref(SrPreferences.srSamplingRate)
        private set
    override var maxSessionSize by corePref(SrPreferences.maxSessionSize)
        private set
    override var maxScreenshotsSizePerSession by corePref(SrPreferences.maxScreenshotsSizePerSession)
        private set

    override var isMonitoringAvailable: Boolean by corePref(SrPreferences.monitoringAvailability)
    override val reproProxyAuthId: Int
        get() = ReproProxyAuthID.SessionReplay

    override var isReproStepsEnabledSDK: Boolean = true
    override val isReproStepsEnabled: Boolean
        get() = isReproStepsEnabledSDK && srEnabled

    override var isReproScreenShotsEnabledSDK: Boolean = true
    override val isReproScreenshotsEnabled: Boolean
        get() = isReproScreenShotsEnabledSDK && srEnabled && this.isReproScreenshotsAvailable

    private var internalSessionLinkPrefix: String by corePref(SrPreferences.sessionLinkPrefix)
    override var isCallbackEnabled: Boolean by corePref(SrPreferences.callbackEnabled)
        private set
    override val sessionLinkPrefix: String?
        get() = internalSessionLinkPrefix
            .takeUnless { content -> content == SR_SESSION_LINK_PREFIX_DEFAULT }

    override fun handle(modesMap: Map<Int, Int>) {
        modesMap[IssueType.SessionReplay]?.let { featureMode ->
            (featureMode > ReproMode.Disable).let { isReproStepsEnabled ->
                reproStepsRTEnabled = isReproStepsEnabled
                this.isReproStepsEnabledSDK = isReproStepsEnabled
            }
            (featureMode > ReproMode.EnableWithNoScreenshots).let { isReproScreenshotsEnabled ->
                screenshotsRTEnabled = isReproScreenshotsEnabled
                this.isReproScreenShotsEnabledSDK = isReproScreenshotsEnabled
            }
        }
    }

    override fun handleConfigurationsChange(newConfig: String) {
        runOrLogError(errorMessage = SOMETHING_WENT_WRONG_WHILE_HANDLING_SESSION_REPLAY_CONFIGURATIONS_CHANGE) {
            newConfig.let(::JSONObject)
                .handleLimitsKeys()
                .handleSRKeys()
        }
    }

    private fun JSONObject.handleSRKeys() = optJSONObject(SR_KEY)?.apply {
        srAvailable = optBoolean(SR_ENABLED_KEY, SR_ENABLED_DEFAULT)
        networkLogsAvailable = optBoolean(SR_NETWORK_KEY, SR_NETWORK_ENABLED_DEFAULT)
        userStepsAvailable = optBoolean(SR_USER_STEPS_KEY, SR_USER_STEPS_ENABLED_DEFAULT)
        this@SRConfigurationsHandlerImpl.isReproScreenshotsAvailable =
            optBoolean(SR_SCREENSHOTS_KEY, SR_SCREENSHOTS_ENABLED_DEFAULT)
        screenshotsCompressionQuality = optDouble(
            SR_SCREENSHOTS_COMPRESSION_KEY,
            SR_SCREENSHOTS_COMPRESSION_DEFAULT.toDouble()
        ).toFloat()
        ibgLogsAvailable = optBoolean(SR_INSTABUG_LOG_KEY, SR_INSTABUG_LOG_DEFAULT)
        maxSDKSize = optDouble(SR_MAX_SDK_MB_KEY, SR_MAX_SDK_SIZE_DEFAULT.toDouble()).toFloat()
        maxLogs = optInt(SR_MAX_LOGS_KEY, SR_MAX_LOGS_DEFAULT)
        samplingRate = optInt(SR_SAMPLING_RATE_KEY, SR_SAMPLING_RATE_DEFAULT)
        maxSessionSize =
            optDouble(SR_MAX_SESSION_MB_KEY, SR_MAX_SESSION_SIZE_DEFAULT.toDouble()).toFloat()
        maxScreenshotsSizePerSession = optDouble(
            SR_MAX_SCREENSHOTS_MP_PER_SESSION_KEY,
            SR_MAX_SCREENSHOTS_SIZE_PER_SESSION_DEFAULT.toDouble()
        ).toFloat()
        isMonitoringAvailable =
            optBoolean(SR_MONITORING_AVAILABILITY_KEY, SR_MONITORING_AVAILABILITY_DEFAULT)
        internalSessionLinkPrefix =
            optString(SR_SESSION_LINK_PREFIX_KEY, SR_SESSION_LINK_PREFIX_DEFAULT)
        isCallbackEnabled = optBoolean(SR_CALLBACK_KEY, SR_CALLBACK_ENABLED_DEFAULT)
    }

    private fun JSONObject.handleLimitsKeys() = apply {
        networkLogLimit = optInt(NETWORK_LOG_LIMIT_KEY, NETWORK_LIMIT_DEFAULT)
        ibgLogsLimit = optInt(INSTABUG_LOG_LIMIT_KEY, INSTABUG_LOG_LIMIT_DEFAULT)
    }

    override fun get(logType: String): Int = when (logType) {
        SRLogType.IBG_LOG -> ibgLogsLimit
        SRLogType.NETWORK_LOG -> networkLogLimit
        else -> Int.MAX_VALUE
    }
}
