package com.instabug.library.networkv2.service.synclogs;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.IBGNetworkWorker;
import com.instabug.library.networkv2.NetworkManager;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestType;
import com.instabug.library.networkv2.service.base.BaseScheduler;
import com.instabug.library.networkv2.service.base.BaseService;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.util.List;

/**
 * A class that perform syncing logs
 */
public class SyncLogService extends BaseService {

    private SyncLogRequestMapper syncLogRequestMapper;

    @Nullable
    private static SyncLogService syncLogService;

    private SyncLogService(@NonNull NetworkManager networkManager, @NonNull SyncLogRequestMapper syncLogRequestMapper,
                           @Nullable Request.Callbacks callbacks,
                           @NonNull BaseScheduler scheduler) {
        super(networkManager, scheduler, callbacks);
        this.syncLogRequestMapper = syncLogRequestMapper;
    }

    public static synchronized SyncLogService getInstance(@NonNull NetworkManager networkManager,
                                                          @NonNull SyncLogRequestMapper syncLogRequestMapper,
                                                          @Nullable Request.Callbacks callbacks,
                                                          @NonNull BaseScheduler scheduler) {
        if (syncLogService == null) {
            syncLogService = new SyncLogService(networkManager, syncLogRequestMapper,
                    callbacks, scheduler);
        }
        return syncLogService;
    }


    /**
     * A method to sync all files,one file per request
     *
     * @param fileList
     * @param uuid
     * @param email
     * @throws UnsupportedEncodingException
     */
    public void sync(List<File> fileList, @Nullable String uuid, @Nullable String email,
                     String appToken) throws UnsupportedEncodingException {
        for (File file : fileList) {
            final Request request = syncLogRequestMapper.getRequest(file, uuid, email, appToken);
            getNetworkManager().doRequest(IBGNetworkWorker.CORE, RequestType.MULTI_PART, request, new Request.Callbacks<RequestResponse, Throwable>() {
                @Override
                public void onSucceeded(RequestResponse response) {
                    if (request.getFileToUpload() != null) {
                        String filePath = request.getFileToUpload().getFilePath();

                        if (getCallbacks() != null) {
                            getCallbacks().onSucceeded(filePath);
                        }
                    }
                }

                @Override
                public void onFailed(Throwable error) {
                    if (getCallbacks() != null) {
                        getCallbacks().onFailed(error);
                    }
                }
            });
        }
    }

    public synchronized static void tearDown() {
        syncLogService = null;
    }

}
