package com.instabug.library.model.v3Session

import com.instabug.library.model.common.Extractable
import com.instabug.library.model.v3Session.StartTimeKeys.BACKGROUND_START_TIME_KEY
import com.instabug.library.model.v3Session.StartTimeKeys.FOREGROUND_START_TIME_KEY
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.providers.BackgroundStateProvider


data class StartTime(
    val startNanoTime: Long,
    val backgroundMicroStartTime: Long,
    val foregroundMicroStartTime: Long,
) : Extractable {
    val isBackground get() = foregroundMicroStartTime == START_TIME_UNDEFINED
    val value
        get() = if (isBackground)
            backgroundMicroStartTime
        else
            foregroundMicroStartTime

    companion object Factory {
        const val START_TIME_UNDEFINED = -1L

        fun create(
            sessionEvent: SessionEvent,
            backgroundStateProvider: BackgroundStateProvider = IBGSessionServiceLocator.backgroundStateProvider
        ): StartTime =
            with(backgroundStateProvider) {
                val startMicroTime = sessionEvent.eventTimeMicro
                val startNanoTime = sessionEvent.eventNanoTime
                return if (isAppStartedInBackground)
                    inBackground(nanoStartTime = startNanoTime, microStartTime = startMicroTime)
                else
                    inForeground(nanoStartTime = startNanoTime, microStartTime = startMicroTime)

            }

        private fun inForeground(nanoStartTime: Long, microStartTime: Long) = StartTime(
            startNanoTime = nanoStartTime,
            foregroundMicroStartTime = microStartTime,
            backgroundMicroStartTime = START_TIME_UNDEFINED
        )


        private fun inBackground(nanoStartTime: Long, microStartTime: Long) = StartTime(
            startNanoTime = nanoStartTime,
            backgroundMicroStartTime = microStartTime,
            foregroundMicroStartTime = START_TIME_UNDEFINED
        )

    }

    override fun extractFields(map: MutableMap<String, Any>): MutableMap<String, Any> =
        map.apply {
            if (foregroundMicroStartTime != START_TIME_UNDEFINED)
                put(FOREGROUND_START_TIME_KEY, foregroundMicroStartTime)
            if (backgroundMicroStartTime != START_TIME_UNDEFINED)
                put(BACKGROUND_START_TIME_KEY, backgroundMicroStartTime)
        }
}
