package com.instabug.library.internal.storage.cache.dbv2;

import android.content.ContentValues;

import androidx.annotation.Nullable;

import com.instabug.library.Feature;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.encryption.EncryptionManager;

import java.util.HashMap;
import java.util.Map;

public class IBGContentValues {

    private final HashMap<String, Object> valuesMap;
    private final boolean isEncryptionEnabled;

    public IBGContentValues() {
        valuesMap = new HashMap<>();
        isEncryptionEnabled = InstabugFeaturesManager.getInstance().getDbEncryptionState() == Feature.State.ENABLED;
    }

    public void put(String key, String value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : value);
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(value, EncryptionManager.IV_V2));
        }
    }

    public void put(String key, Byte value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }

    public void put(String key, Short value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }

    public void put(String key, Long value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }

    public void put(String key, Integer value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }

    public void put(String key, Float value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }


    public void put(String key, Double value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }

    public void put(String key, Boolean value, boolean transitive) {
        if (transitive || !isEncryptionEnabled) {
            valuesMap.put(key, value == null ? null : String.valueOf(value));
        } else {
            valuesMap.put(key, EncryptionManager.encrypt(String.valueOf(value), EncryptionManager.IV_V2));
        }
    }

    @Nullable
    public Object get(String key) {
        return valuesMap.get(key);
    }


    public ContentValues toContentValues() {
        ContentValues contentValues = new ContentValues();
        for (Map.Entry<String, Object> entry : valuesMap.entrySet()) {
            if (entry.getValue() != null) {
                contentValues.put(entry.getKey(), String.valueOf(entry.getValue()));
            }
        }

        return contentValues;
    }
}
