package com.instabug.library.internal.filestore

import androidx.annotation.VisibleForTesting

fun interface SpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory> :
    DirectorySelector<In, SpanDir>

/**
 * Selects the current span if and only if the representing directory exists
 */
class ActiveCurrentSpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory> :
    SpanSelector<In, SpanDir> {
    override fun invoke(input: In): SpanDir? = input.currentDirectory?.takeIf { it.exists() }
}

/**
 * Selects the current span directory whether exits or not.
 */
class CurrentSpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory> :
    SpanSelector<In, SpanDir> {
    override fun invoke(input: In): SpanDir? = input.currentDirectory
}

/**
 * Selects an old span given a [spanId].
 * The output directory is guaranteed to exist implicitly.
 */
class OldSpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory>(
    private val spanId: String
) : SpanSelector<In, SpanDir> {
    override fun invoke(input: In): SpanDir? =
        input.oldDirectories
            .find { spanDirectory -> spanDirectory.name == spanId }
}

/**
 * Selects a span given it's [spanId] whether it's the current or an old span.
 */
class MatchingIDSpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory>(
    @get:VisibleForTesting val spanId: String
) : SpanSelector<In, SpanDir> {
    override fun invoke(input: In): SpanDir? =
        input.currentDirectory?.takeIf { current -> current.name == spanId }
            ?: OldSpanSelector<In, SpanDir>(spanId).invoke(input)
}

/**
 * Selects the oldest span if and only if the number of old spans exceeds the given [limit]
 */
class OldestSpanOnLimitSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory>(
    private val limit: Int
) : SpanSelector<In, SpanDir> {
    override fun invoke(input: In): SpanDir? =
        input.oldDirectories
            .takeIf { directories -> directories.size > limit }
            ?.minByOrNull { directory -> directory.lastModified() }
}

fun interface MultiSpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory> :
    MultiDirectorySelector<In, SpanDir>

/**
 * Selects multiple spans given their [idsList]
 */
class IDsMatchMultiSpanSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory>(
    private val idsList: List<String>
) : MultiSpanSelector<In, SpanDir> {
    override fun invoke(input: In): List<SpanDir> =
        input.oldDirectories.filter { spanDir -> idsList.contains(spanDir.name) }
}

/**
 * Selects all spans including the current one.
 */
class AllSpansSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory> :
    MultiSpanSelector<In, SpanDir> {
    override fun invoke(input: In): List<SpanDir> =
        with(input) { oldDirectories.toMutableList().apply { currentDirectory?.let(this::add) } }
}

/**
 * Selects all spans excluding the current one (old spans only).
 */
class OldSpansSelector<In : SpansDirectory<SpanDir>, SpanDir : Directory> :
    MultiSpanSelector<In, SpanDir> {
    override fun invoke(input: In): List<SpanDir> = input.oldDirectories
}