package com.instabug.library.experiments;

import static com.instabug.library.experiments.constants.Constants.EXPERIMENTS_STORE_LIMIT_FALLBACK;
import static com.instabug.library.util.collections.CollectionKtxKt.unique;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.experiments.cache.ExperimentsDBManager;
import com.instabug.library.experiments.constants.Constants;
import com.instabug.library.experiments.constants.ErrorMessages;
import com.instabug.library.experiments.di.ServiceLocator;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;


public class ExperimentsManagerImpl implements ExperimentsManager {

    private static final String EXP_EXECUTOR_NAME = "Experiments";

    @Override
    public void addExperiments(@NonNull final List<String> experiments) {
        if (experiments == null || experiments.isEmpty()) {
            return;
        }
        PoolProvider.postOrderedIOTask(EXP_EXECUTOR_NAME, () -> {
            final int experimentsStoreLimit = getExperimentsStoreLimit();
            List<String> experimentsList = dropDuplicateExperiments(experiments);
            experimentsList = dropInvalidLengthExperiments(experimentsList);
            if (experimentsList.isEmpty()) return;

            int droppedCount = calculateDroppedExperiments(experimentsList, experimentsStoreLimit);
            if (droppedCount != 0)
                experimentsList = experimentsList
                        .subList(droppedCount, experimentsList.size());

            ExperimentsDBManager experimentsDBManager = ServiceLocator.getExperimentsDBManager();
            experimentsDBManager.insertExperiments(experimentsList);
            droppedCount += experimentsDBManager.trimToLimit(experimentsStoreLimit);
            if (droppedCount > 0) {
                InstabugSDKLogger.w(
                        Constants.LOG_TAG,
                        String.format(
                                ErrorMessages.MAX_EXPERIMENTS_COUNT_REACHED,
                                experimentsStoreLimit
                        )
                );
            }
        });
    }

    private static int calculateDroppedExperiments(List<String> experimentsList, int experimentsStoreLimit) {
        return experimentsList.size() > experimentsStoreLimit ?
                (experimentsList.size() - experimentsStoreLimit)
                : 0;
    }

    @Override
    public void removeExperiments(@NonNull final List<String> experiments) {
        if (experiments == null || experiments.isEmpty()) {
            return;
        }
        PoolProvider.postOrderedIOTask(EXP_EXECUTOR_NAME, () -> {
            List<String> experimentsList = dropDuplicateExperiments(experiments);
            ExperimentsDBManager experimentsDBManager = ServiceLocator.getExperimentsDBManager();
            experimentsDBManager.deleteExperiments(experimentsList);
        });
    }

    @Override
    public void clearAllExperiments() {
        PoolProvider.postOrderedIOTask(EXP_EXECUTOR_NAME, () -> {
            ExperimentsDBManager experimentsDBManager = ServiceLocator.getExperimentsDBManager();
            experimentsDBManager.clearAllExperiments();
        });
    }

    @Nullable
    @Override
    public List<String> getExperiments(float percentage) {
        return ServiceLocator.getExperimentsDBManager().getExperiments(percentage);
    }

    @Override
    public void updateExperimentsFeature(@NonNull JSONObject featuresJsonObject) {
        if (featuresJsonObject != null) {
            int storeLimit = featuresJsonObject.optInt("experiments_limit", EXPERIMENTS_STORE_LIMIT_FALLBACK);
            SettingsManager settingsManager = SettingsManager.getInstance();
            if (settingsManager != null) {
                settingsManager.setExperimentsStoreLimit(storeLimit);
            }
        }
    }

    private List<String> dropDuplicateExperiments(List<String> experiments) {
        return new ArrayList<>(unique(experiments, true));
    }

    private List<String> dropInvalidLengthExperiments(List<String> experiments) {
        List<String> experimentsList = new ArrayList<>();
        boolean isMaxLengthMet = false;
        for (String experiment : experiments) {
            if (experiment.trim().length() > Constants.EXPERIMENT_MAX_LENGTH) {
                isMaxLengthMet = true;
            } else {
                experimentsList.add(experiment.trim());
            }
        }

        if (isMaxLengthMet) {
            InstabugSDKLogger.e(Constants.LOG_TAG, ErrorMessages.MAX_EXPERIMENT_LENGTH_MET);
        }
        return experimentsList;
    }

    private int getExperimentsStoreLimit() {
        int experimentsLimit = Constants.EXPERIMENTS_STORE_LIMIT_FALLBACK;
        SettingsManager settingsManager = SettingsManager.getInstance();
        if (settingsManager != null) {
            experimentsLimit = settingsManager.getExperimentsStoreLimit();
        }
        return CoreServiceLocator.getLimitConstraintApplier().applyConstraints(experimentsLimit);
    }
}
