package com.instabug.library.util.threading;

import android.os.Looper;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class ThreadUtils {

    /**
     * @return true if the current thread is the Main thread
     */
    public static boolean isCurrentThreadMain() {
        return Looper.myLooper() == Looper.getMainLooper();
    }

    /**
     * @param errorName
     * @return the stacktrace of the main thread
     */
    private static String getMainThreadStackTrace(String errorName) {
        Thread mainThread = Looper.getMainLooper().getThread();
        return buildStackTraceFromElements(mainThread.getStackTrace(), errorName);
    }

    /**
     * @param stackTraceElements The stacktrace elements of a thread
     * @param errorName
     * @return the stacktrace for the thread represented as a {@link String}
     */
    public static String buildStackTraceFromElements(StackTraceElement[] stackTraceElements, @Nullable String errorName) {
        StringBuilder stackTraceBuilder = new StringBuilder();
        if (errorName != null) {
            stackTraceBuilder.append(errorName).append("\n");
        }
        for (StackTraceElement stackTraceElement : stackTraceElements) {
            stackTraceBuilder.append("\t at ");
            stackTraceBuilder.append(stackTraceElement.toString());
            stackTraceBuilder.append("\n");
        }
        return stackTraceBuilder.toString();
    }

    public static JSONObject getMainThreadData(@Nullable String errorName) throws JSONException {
        Looper looper = Looper.getMainLooper();
        Thread thread = looper.getThread();
        JSONObject mainThreadData = getThreadData(thread);
        mainThreadData.put("error", getStackTraceAsError(Looper.getMainLooper().getThread(), errorName));
        return mainThreadData;
    }

    private static JSONObject getStackTraceAsError(Thread thread, @Nullable String errorName) {
        JSONObject exception = new JSONObject();

        try {
            if (errorName != null) {
                exception.put("name", "ANRError: Application Not Responding for at least 5000 ms.\n");
                exception.put("exception", errorName);
            }
            StackTraceElement element = null;
            thread.getStackTrace();
            if (thread.getStackTrace().length > 0) {
                element = thread.getStackTrace()[0];
            }

            if (element != null && element.getFileName() != null) {
                exception.put("location", element.getFileName() + ":" + element.getLineNumber());
            }

            String error = errorName == null ? "" : errorName;
            exception.put("stackTrace", getMainThreadStackTrace(error));
        } catch (JSONException jsonException) {
            InstabugSDKLogger.e(Constants.LOG_TAG, jsonException.getMessage() != null
                    ? jsonException.getMessage()
                    : "Json exception while creating formatted exception", jsonException);
        }
        return exception;
    }

    public static JSONObject getThreadData(Thread thread) throws JSONException {
        JSONObject threadJsonObject = new JSONObject();
        JSONObject t = new JSONObject();
        t.put("threadName", thread.getName());
        t.put("threadId", thread.getId());
        t.put("threadPriority", thread.getPriority());
        t.put("threadState", thread.getState().toString());
        ThreadGroup group = thread.getThreadGroup();
        if (group != null) {
            JSONObject threadGroup = new JSONObject();
            threadGroup.put("name", group.getName());
            threadGroup.put("maxPriority", group.getMaxPriority());
            threadGroup.put("activeCount", group.activeCount());
            t.put("threadGroup", threadGroup);
        }

        threadJsonObject.put("thread", t);

        return threadJsonObject;
    }

    private static boolean isMainThread(Thread thread) {
        return Looper.getMainLooper()!= null && thread == Looper.getMainLooper().getThread();
    }

    public static JSONArray getThreadsData(@Nullable Thread crashingThread) throws JSONException {
        JSONArray threadsData = new JSONArray();
        for (Thread thread : ThreadsProvider.getAllStackTraces().keySet()) {
            JSONObject threadData = getThreadData(thread);
            ((JSONObject) threadData.get("thread")).put("isMain", isMainThread(thread));
            ((JSONObject) threadData.get("thread")).put("stackTrace", getStackTraceFromThread(thread));
            ((JSONObject) threadData.get("thread")).put("isCrashing", thread == crashingThread);
            threadsData.put(threadData);
        }
        return threadsData;
    }

    public static String getStackTraceFromThread(Thread thread) {
        StringBuilder stackTraceBuilder = new StringBuilder();
        for (StackTraceElement stackTraceElement : thread.getStackTrace()) {
            stackTraceBuilder.append("\t at ");
            stackTraceBuilder.append(stackTraceElement.toString());
            stackTraceBuilder.append("\n");
        }
        return stackTraceBuilder.toString();
    }

    public static JSONObject getThreadDataAndStackTrace(Thread thread, Throwable throwable) throws JSONException {
        JSONObject threadJsonObject = new JSONObject();
        threadJsonObject.put("crashed", true);
        threadJsonObject.put("threadId", thread.getId());
        threadJsonObject.put("isMain", isMainThread(thread));
        threadJsonObject.put("stackTrace", throwable.getStackTrace());
        return threadJsonObject;
    }

    public static StackTraceElement[] getCurrentThreadStackTrace() {
        return Thread.currentThread().getStackTrace();
    }
}
