package com.instabug.library.util;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.LocaleList;
import android.text.TextUtils;
import android.view.View;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.settings.SettingsManager;

import java.util.Locale;

/**
 * @author mSobhy
 */
public class LocaleUtils {

    public static void setLocale(Activity activity, Locale locale) {
        Configuration configuration = activity.getResources()
                .getConfiguration();

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            if (locale != null) {
                configuration.setLocales(new LocaleList(locale));
                updateResources(activity, locale);
            }
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            configuration.setLocale(locale);
        } else {
            configuration.locale = locale;
        }

        activity.getResources()
                .updateConfiguration(configuration, activity.getResources().getDisplayMetrics());
    }

    public static void setAppLocale(Activity activity) {
        Locale appLocale = SettingsManager.getInstance().getAppLocale();
        if (appLocale != null) {
            setLocale(activity, appLocale);
        }
    }

    @TargetApi(Build.VERSION_CODES.N)
    private static Context updateResources(Context context, Locale locale) {
        Locale.setDefault(locale);
        Configuration configuration = context.getResources().getConfiguration();
        configuration.setLocale(locale);
        configuration.setLayoutDirection(locale);
        return context.createConfigurationContext(configuration);
    }

    /**
     * Returns local string resource in desired locale
     *
     * @param requestedLocale
     * @param resourceId
     * @param context
     * @return localized string resource
     */
    public static String getLocaleStringResource(Locale requestedLocale, int resourceId, @Nullable Context context) {
        return getLocaleStringResource(requestedLocale, resourceId, context, null);
    }

    /**
     * Returns local string resource in desired locale formatted in the desired format
     *
     * @param requestedLocale
     * @param resourceId
     * @param context
     * @param formatArgs
     * @return
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    public static String getLocaleStringResource(Locale requestedLocale, int resourceId, @Nullable Context context, @Nullable Object... formatArgs) {
        String result;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            result = getLocalizedStringResNewAPI(context, resourceId, requestedLocale, formatArgs);
            if (result == null) {
                result = getLocalizedStringResOldAPI(context, resourceId, requestedLocale, formatArgs);
                if (result == null) {
                    // fallback to the original string if localization failed
                    if (context != null) {
                        result = context.getString(resourceId);
                        if (formatArgs != null) {
                            result = String.format(result, formatArgs);
                        }
                    } else {
                        return "";
                    }
                }
            }
        } else {
            result = getLocalizedStringResOldAPI(context, resourceId, requestedLocale, formatArgs);
        }
        return result;
    }

    @Nullable
    @TargetApi(Build.VERSION_CODES.JELLY_BEAN_MR1)
    private static String getLocalizedStringResNewAPI(@Nullable Context context, int resourceId, Locale requestedLocale, @Nullable Object... formatArgs) {
        try {
            if (context != null) {
                Configuration config = new Configuration(context.getResources().getConfiguration());
                config.setLocale(requestedLocale);
                Context configurationContext = context.createConfigurationContext(config);
                String result = configurationContext == null ? null : configurationContext.getText(resourceId).toString();
                if (formatArgs != null && result != null) {
                    result = String.format(result, formatArgs);
                }
                return result;
            } else {
                return null;
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e.getMessage() + " while getting localized string");
            return null;
        }
    }

    @Nullable
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private static String getLocalizedStringResOldAPI(@Nullable Context context, int resourceId, Locale requestedLocale, @Nullable Object... formatArgs) {
        if (context != null) {
            Resources resources = context.getResources();
            Configuration conf = resources.getConfiguration();
            Locale savedLocale = conf.locale;
            conf.locale = requestedLocale;
            resources.updateConfiguration(conf, null);
            // retrieve resources from desired locale
            String result = resources.getString(resourceId);
            // restore original locale
            conf.locale = savedLocale;
            resources.updateConfiguration(conf, null);
            if (formatArgs != null) {
                result = String.format(result, formatArgs);
            }
            return result;
        }
        return null;
    }

    public static boolean isRTL(Locale locale) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            return TextUtils.getLayoutDirectionFromLocale(locale) == View.LAYOUT_DIRECTION_RTL;
        } else {
            final int directionality = Character.getDirectionality(locale.getDisplayName().charAt(0));
            return directionality == Character.DIRECTIONALITY_RIGHT_TO_LEFT ||
                    directionality == Character.DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC;
        }
    }

    /**
     * Returns the current locale identifier as defined in IBGSRV-5481 if the current locale
     * is one of the support languages by the SDK or falls back to the default
     *
     * @param context a non-ui context
     * @return the current locale identifier if supported by the SDK or "default" as a fallback
     */
    public static String getCurrentLocaleResolved(Context context) {
        return new SdkLocaleRegistry(context).getCurrentLocaleResolved();
    }

    /**
     * Returns the passed locale identifier as defined in IBGSRV-5481 if it is one of the support
     * languages by the SDK or falls back to the default
     *
     * @param context a non-ui context
     * @param locale  the locale do resolved and extract the identifier
     * @return the identifier if the locale is supported by the SDK or "default" as a fallback
     */
    public static String resolveLocale(Context context, Locale locale) {
        return new SdkLocaleRegistry(context).resolveLocale(locale);
    }

    public static boolean isSupportedBySdk(Context context, Locale locale) {
        if (context == null || locale == null) {
            return false;
        }
        return new SdkLocaleRegistry(context).isSupported(locale.getLanguage());
    }
}
