package com.instabug.library.sessionreplay

import android.net.Uri
import android.webkit.MimeTypeMap
import com.instabug.library.IBGNetworkWorker
import com.instabug.library.InstabugNetworkJob
import com.instabug.library.networkv2.INetworkManager
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.limitation.RateLimitationSettingsImpl
import com.instabug.library.networkv2.limitation.RateLimitedFeature
import com.instabug.library.networkv2.limitation.RateLimiter
import com.instabug.library.networkv2.request.Endpoints
import com.instabug.library.networkv2.request.FileToUpload
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestMethod
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.extenstions.logError
import com.instabug.library.util.extenstions.logVerbose
import java.io.File

private const val DEFAULT_FILE_TYPE = ""

typealias CompositeSession = Pair<SRSessionMetadata, SRSessionDirectory>

class SRSyncJob(
    private val networkManager: INetworkManager,
    private val metadataHandler: SRMetadataDBHandler,
    private val filesDirectory: OperableSpansCacheDirectory<SRSessionDirectory>,
    private val configurations: SRConfigurationsProvider,
    private val rateLimiter: RateLimiter<SRSessionMetadata, RateLimitationSettingsImpl> =
        RateLimiter.Factory.createWithDefaultSettings(RateLimitedFeature.SESSION_REPLAY) { metadata ->
            DeleteSessionDirectoryOperation(metadata.uuid).also(filesDirectory::operate)
            metadataHandler.delete(metadata.uuid)
        }
) : InstabugNetworkJob() {
    override fun start() {
        enqueueJob(IBGNetworkWorker.CORE) {
            metadataHandler
                .takeIf { configurations.srEnabled }
                ?.queryByStatus(SyncStatus.READY_FOR_SYNC, SyncStatus.READY_FOR_SCREENSHOTS_SYNC)
                ?.asSequence()
                ?.mapNotNull(this::toCompositeSession)
                ?.onEach(this::fireLogsRequest)
                ?.forEach(this::fireScreenshotsRequest)
        }
    }

    private fun toCompositeSession(metadata: SRSessionMetadata): CompositeSession? = runCatching {
        filesDirectory.operate(GetSessionDirectoryOperation(metadata.uuid)).get()
            ?.takeIf { dir -> dir.compressedLogsFile.exists() || dir.screenshotsZipFile.exists() }
            ?.let { dir -> CompositeSession(metadata, dir) }
            ?: run { handleSessionWithMissingLogs(metadata); null }
    }.getOrNull()

    private fun handleSessionWithMissingLogs(metadata: SRSessionMetadata) {
        "No logs or screenshots found for session ${metadata.uuid}, deleting...".logVerbose()
        metadataHandler.delete(metadata.uuid)
        filesDirectory.operate(DeleteSessionDirectoryOperation(metadata.uuid))
    }

    private fun fireLogsRequest(composite: CompositeSession) {
        if (rateLimiter.applyIfPossible(composite.first)) return
        val (metadata, directory) = composite
        if (metadata.status != SyncStatus.READY_FOR_SYNC) return
        if (!directory.compressedLogsFile.exists()) {
            logsSuccessActions(metadata, directory)
            return
        }
        val callback = object : Request.Callbacks<RequestResponse, Throwable> {
            override fun onSucceeded(response: RequestResponse?) {
                "Replay logs for session ${metadata.uuid} sent successfully".logVerbose()
                logsSuccessActions(metadata, directory)
                "Replay logs file for session ${metadata.uuid} deleted".logVerbose()
                rateLimiter.reset()
            }

            override fun onFailed(error: Throwable?) {
                if (error?.let { rateLimiter.inspect(it, metadata) } == true) return
                "Failed to send replay logs for session ${metadata.uuid}".logError(error)
            }
        }
        val fileGetter = { dir: SRSessionDirectory -> dir.compressedLogsFile }
        SRRequestProvider.createFileUploadRequest(
            composite,
            fileGetter,
            Endpoints.SESSION_LOGS
        )?.let { request ->
            networkManager.doRequestOnSameThread(RequestType.MULTI_PART, request, callback)
        }
    }

    private fun logsSuccessActions(metadata: SRSessionMetadata, directory: SRSessionDirectory) {
        metadataHandler.updateStatus(metadata.uuid, SyncStatus.READY_FOR_SCREENSHOTS_SYNC)
        metadata.status = SyncStatus.READY_FOR_SCREENSHOTS_SYNC
        directory.compressedLogsFile.takeIf(File::exists)?.delete()
    }

    private fun fireScreenshotsRequest(composite: CompositeSession) {
        val (metadata, directory) = composite
        if (metadata.status != SyncStatus.READY_FOR_SCREENSHOTS_SYNC) return
        if (!directory.screenshotsZipFile.exists()) {
            screenshotsSuccessActions(metadata, directory)
            return
        }
        val callback = object : Request.Callbacks<RequestResponse, Throwable> {
            override fun onSucceeded(response: RequestResponse?) {
                "Replay screenshots for session ${metadata.uuid} sent successfully".logVerbose()
                screenshotsSuccessActions(metadata, directory)
                "Replay dir & metadata for session ${metadata.uuid} deleted".logVerbose()
            }

            override fun onFailed(error: Throwable?) {
                "Failed to send replay screenshots for session ${metadata.uuid}".logError(error)
            }
        }
        val fileGetter = { dir: SRSessionDirectory -> dir.screenshotsZipFile }
        SRRequestProvider.createFileUploadRequest(
            composite,
            fileGetter,
            Endpoints.SESSION_SCREENSHOTS
        )?.let { request ->
            networkManager.doRequestOnSameThread(RequestType.MULTI_PART, request, callback)
        }
    }

    private fun screenshotsSuccessActions(
        metadata: SRSessionMetadata,
        directory: SRSessionDirectory
    ) {
        metadataHandler.updateStatus(metadata.uuid, SyncStatus.SYNCED)
        directory.deleteFilesDirectory()
        metadataHandler.delete(metadata.uuid)
    }
}

object SRRequestProvider {
    fun createFileUploadRequest(
        composite: CompositeSession,
        fileGetter: (SRSessionDirectory) -> File,
        endpoint: String
    ): Request? {
        val (metadata, directory) = composite
        val sessionCompositeId =
            "${SettingsManager.getInstance().appToken}-${metadata.startTime}-${metadata.partialId}"
        val finalizedEndpoint = endpoint.replace(":session_id".toRegex(), sessionCompositeId)
        val (filePath, fileName) = runCatching {
            Uri.parse(fileGetter(directory).absolutePath)
                .let { uri -> (uri.path to uri.lastPathSegment) }
        }.getOrDefault(null to null)
        if (filePath == null || fileName == null) return null
        val fileType = MimeTypeMap.getFileExtensionFromUrl(fileName)
            ?.let { extension -> MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension) }
            ?: DEFAULT_FILE_TYPE
        return Request.Builder()
            .method(RequestMethod.POST)
            .url(finalizedEndpoint)
            .type(RequestType.MULTI_PART)
            .fileToUpload(FileToUpload("file", fileName, filePath, fileType))
            .build()
    }
}