package com.instabug.library.sessionV3.sync

import androidx.annotation.VisibleForTesting
import com.instabug.library.model.v3Session.IBGSessionDTO
import com.instabug.library.model.v3Session.IBGSessionKeys
import com.instabug.library.model.v3Session.IBGSessionMap
import com.instabug.library.model.v3Session.RatingDialogDetection

/**
 * merges sessions maps and extract common keys to be in the batch root request
 * and Unique keys to be in the session level
 * */
interface IBGSessionMerger {
    /**
     * @param keys these keys will be ignored while merging the sessions map to extract common
     *
     * */
    fun addUniqueKeys(keys: Iterable<String>)

    /**
     * merge sessions to extract common keys to send them in the batch root request
     *
     * @param sessions a list of sessions map that will be merged
     * @return session DTO that contains common keys , unique keys , sessions ids
     * @see  IBGSessionDTO
     * */
    fun mergeCommon(sessions: List<IBGSessionMap>): IBGSessionDTO
}

object IBGSessionMergerImpl : IBGSessionMerger {

    private val uniqueKeys = mutableSetOf(
        IBGSessionKeys.ID_KEY,
        IBGSessionKeys.RANDOM_PARTIAL_SESSION_ID,
        RatingDialogDetection.REVIEW_PROMPT_KEY
    )

    override fun addUniqueKeys(keys: Iterable<String>) {
        uniqueKeys.addAll(keys)
    }

    override fun mergeCommon(sessions: List<IBGSessionMap>): IBGSessionDTO = sessions
        .let { sessionsList ->
            val commonKeys = extractCommon(sessionsList)
            val sessionsWithoutCommonKeys = sessionsList.extractUnique(commonKeys)
            val sessionsIds = sessions.mapNotNull { it[IBGSessionKeys.ID_KEY] as String? }
            IBGSessionDTO(
                commonKeys = commonKeys,
                sessions = sessionsWithoutCommonKeys,
                sessionsIds = sessionsIds
            )
        }

    private fun List<IBGSessionMap>.extractUnique(
        commonKeys: Map<String, Any>
    ) = map { session -> removeCommonKeys(session, commonKeys) }

    private fun removeCommonKeys(
        session: IBGSessionMap,
        commonKeys: Map<String, Any>
    ) = session.filterKeys { key -> !commonKeys.containsKey(key) }

    private fun extractKeys(sessions: List<IBGSessionMap>) = sessions
        .flatMap { session -> session.keys }
        .distinct()

    private fun extractCommon(sessions: List<IBGSessionMap>): Map<String, Any> =
        extractKeys(sessions)
            .mapNotNull { key -> getCommonKeyOrNull(sessions, key) }
            .toMap()

    private fun getCommonKeyOrNull(
        sessions: List<IBGSessionMap>,
        key: String
    ): Pair<String, Any>? {
        val values = values(key, sessions)
        return if (!uniqueKeys.contains(key) && isCommon(values, sessions))
            key to values.first()!!
        else
            null
    }

    private fun values(
        key: String,
        sessions: List<IBGSessionMap>,
    ) = sessions.map { session -> session[key] }

    private fun isCommon(
        values: List<Any?>,
        sessions: List<IBGSessionMap>
    ) = values.distinct().size == 1 && values.size == sessions.size

    @VisibleForTesting
    fun resetKeys() {
        uniqueKeys.apply {
            clear()
            add(IBGSessionKeys.ID_KEY)
            add(IBGSessionKeys.RANDOM_PARTIAL_SESSION_ID)
            add(RatingDialogDetection.REVIEW_PROMPT_KEY)
        }
    }
}