package com.instabug.library.session;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.device.InstabugDeviceProperties;
import com.instabug.library.internal.utils.PreferencesUtils;
import com.instabug.library.model.session.CoreSession;
import com.instabug.library.model.session.SessionLocalEntity;
import com.instabug.library.model.session.SessionMapper;
import com.instabug.library.model.session.SessionsBatchDTO;
import com.instabug.library.model.session.config.SessionsConfig;
import com.instabug.library.model.session.config.SyncMode;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.TimeUtils;

import org.jetbrains.annotations.NotNull;

import java.util.List;
import java.util.concurrent.TimeUnit;

public class SessionsSyncManager {

    private SessionsConfig config;
    @NonNull
    private final SessionBatcher sessionBatcher;
    @NonNull
    private final PreferencesUtils preferencesUtils;
    @NonNull
    private final SessionsLocalDataSource localDataSource;
    @NonNull
    private final SessionsRemoteDataSource remoteDataSource;
    @NonNull
    private final SessionsConfigurationsManager configurationsManager;

    public final static String SYNCING_SESSIONS_ERROR = "Syncing Sessions filed due to: ";

    public static SessionsSyncManager create(@NonNull Context context) {
        SessionBatcher batcher = new SessionsBatcherImpl();
        PreferencesUtils preferencesUtils = SessionsServiceLocator.getPreferencesUtils(context);
        SessionsLocalDataSource localDataSource = SessionsServiceLocator.getSessionsLocalDataSource();
        SessionsRemoteDataSource remoteDataSource = SessionsRemoteDataSource.getInstance(context);
        SessionsConfig config = SettingsManager.getSessionsSyncConfigurations(context);
        SessionsConfigurationsManager configurations = SessionsServiceLocator.getSessionsConfigurations();
        return new SessionsSyncManager(config, batcher, preferencesUtils, localDataSource, remoteDataSource, configurations);
    }

    public SessionsSyncManager(@NonNull SessionsConfig config,
                               @NonNull SessionBatcher sessionBatcher,
                               @NonNull PreferencesUtils preferencesUtils,
                               @NonNull SessionsLocalDataSource localDataSource,
                               @NonNull SessionsRemoteDataSource remoteDataSource, @NonNull SessionsConfigurationsManager configurationsManager) {
        this.config = config;
        this.sessionBatcher = sessionBatcher;
        this.preferencesUtils = preferencesUtils;
        this.localDataSource = localDataSource;
        this.remoteDataSource = remoteDataSource;
        this.configurationsManager = configurationsManager;
    }

    public void onConfigurationsChanged(@NonNull SessionsConfig config) {
        this.config = config;
    }

    @WorkerThread
    public SessionsSyncManager evaluateSessions() {
        long elapsedTime = getElapsedTimeSinceLastIntervalResetInMinutes();
        if (config.getSyncMode() == SyncMode.OFF) {
            logD("Skipping sync. Sync mode = " + config.getSyncMode());
            return this;
        } else if (shouldSync() || config.getSyncMode() == SyncMode.NO_BATCHING) {
            logD("Evaluating cached sessions. Elapsed time since last sync = " + elapsedTime + " mins. Sync configs = " + config.toString());
            deleteInvalidSessions();
            localDataSource.markOfflineAsReadyForSync();
            resetInterval();
        } else if (InstabugDeviceProperties.getVersionCode() != SettingsManager.getInstance().getLastKnownVersionCode()) {
            SettingsManager.getInstance().setVersionCode(InstabugDeviceProperties.getVersionCode());
            SettingsManager.getInstance().setIsFirstSession(true);
            logD("App version has changed. Marking cached sessions as ready for sync");
            localDataSource.markOfflineAsReadyForSync();
        } else {
            logD("Skipping sessions evaluation. Elapsed time since last sync = " + elapsedTime + " mins. Sync configs = " + config.toString());
        }

        return this;
    }

    private void deleteInvalidSessions() {
        if (config.getSyncMode() == SyncMode.DECOMMISSION) {
            logD("deleting invalid session with s2s false ");
            localDataSource.deleteInvalidSessions();
        }
    }

    /**
     * Checks if the sync interval passed
     *
     * @return true if last sync time >= sync interval, false otherwise
     */
    public boolean shouldSync() {
        long elapsedTime = getElapsedTimeSinceLastIntervalResetInMinutes();
        int syncIntervals = config.getSyncIntervalsInMinutes();
        return elapsedTime >= syncIntervals;
    }

    @WorkerThread
    public void sync() {
        if (config.getSyncMode() == SyncMode.OFF) {
            logD("Sessions sync is not allowed. Sync mode = " + config.getSyncMode());
        } else {
            logD("Syncing local with remote. Sync configs = " + config.toString());
            List<SessionLocalEntity> sessionLocalEntities = localDataSource.queryAllReadyForSync();
            if (!sessionLocalEntities.isEmpty()) {
                List<CoreSession> sessions = SessionMapper.toModels(sessionLocalEntities);
                List<SessionsBatchDTO> sessionsBatches;
                if (config.getSyncMode() == SyncMode.NO_BATCHING) {
                    sessionsBatches = sessionBatcher.batch(sessions, 1);
                    logD("Syncing " + sessionsBatches.size() + " batches of max 1 session per batch.");
                } else {
                    int maxSessionsPerRequest = config.getMaxSessionsPerRequest();
                    sessionsBatches = sessionBatcher.batch(sessions, maxSessionsPerRequest);
                    logD("Syncing " + sessionsBatches.size() + " batches of max " + maxSessionsPerRequest + " sessions per batch.");
                }
                syncWithRemote(sessionsBatches);
            } else {
                logD("No sessions ready for sync. Skipping...");
            }
        }
    }

    private void syncWithRemote(@NonNull List<SessionsBatchDTO> batches) {
        SessionsServiceLocator.postNetworkTask(() -> {
            for (final SessionsBatchDTO batch : batches) {
                if (configurationsManager.isSessionsRateLimited()) {
                    handleRateIsLimited(batch);
                } else {
                    configurationsManager.setLastSessionsRequestStartedAt(System.currentTimeMillis());
                    syncBatch(batch);
                }
            }
        });
    }

    private void handleRateIsLimited(@NonNull SessionsBatchDTO batch) {
        List<String> ids = SessionMapper.toIDs(batch);
        localDataSource
                .markAsSyncedWithRemote(ids)
                .delete(ids);
        logRateIsLimited();
    }

    private void syncBatch(@NonNull SessionsBatchDTO batch) {
        List<String> ids = SessionMapper.toIDs(batch);
        remoteDataSource.saveOrUpdate(batch, new Request.Callbacks<RequestResponse, Throwable>() {
            @Override
            public void onSucceeded(RequestResponse response) {
                logD("Synced a batch of " + batch.getSessions().size() + " session/s.");
                configurationsManager.setLastSessionsRequestStartedAt(0);
                localDataSource
                        .markAsSyncedWithRemote(ids)
                        .delete(ids);
            }

            @Override
            public void onFailed(Throwable error) {
                if (error instanceof RateLimitedException) {
                    handleRateLimitedException((RateLimitedException) error, batch);
                } else
                    IBGDiagnostics.reportNonFatalAndLog(error, SYNCING_SESSIONS_ERROR + error.getMessage(), Constants.LOG_TAG);
            }
        });
    }

    private void handleRateLimitedException(RateLimitedException exception, @NonNull SessionsBatchDTO batch) {
        configurationsManager.setSessionsLimitedUntil(exception.getPeriod());
        handleRateIsLimited(batch);
    }

    private void resetInterval() {
        setLastSyncTime(TimeUtils.currentTimeMillis());
    }

    public void setLastSyncTime(long lastSyncTime) {
        preferencesUtils.saveOrUpdateLong("key_last_batch_synced_at", lastSyncTime);
    }

    /**
     * Set the last sync time to a value (Now - syncInterval time) that allows an immediate sync
     * by making {@link  #getElapsedTimeSinceLastIntervalResetInMinutes} >=  {@link SessionsConfig#getSyncIntervalsInMinutes}
     *
     * @see #shouldSync
     */
    public SessionsSyncManager pushSyncInterval() {
        long value = TimeUtils.currentTimeMillis() -
                TimeUnit.MINUTES.toMillis(config.getSyncIntervalsInMinutes());
        setLastSyncTime(value);
        return this;
    }

    private long getElapsedTimeSinceLastIntervalResetInMinutes() {
        long elapsedTimeMillis = TimeUtils.currentTimeMillis() - preferencesUtils.getLong("key_last_batch_synced_at");
        return TimeUnit.MILLISECONDS.toMinutes(elapsedTimeMillis);
    }

    private void logRateIsLimited() {
        logD(String.format(RateLimitedException.RATE_LIMIT_REACHED, Constants.FEATURE_NAME));
    }

    private void logD(@NonNull String message) {
        InstabugSDKLogger.d(Constants.LOG_TAG, message);
    }


    public void setConfig(@NotNull SessionsConfig config) {
        this.config = config;
    }
}
