package com.instabug.library.networkv2.authorization;

import android.util.Base64;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.zip.GZIPOutputStream;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

/**
 * A class that manipulate the data by hashing or encoding or gziping
 */
public final class DataManipulationUtility {


    public static byte[] gzip(String string) throws IOException, OutOfMemoryError {
        ByteArrayOutputStream os = new ByteArrayOutputStream(string.length());
        GZIPOutputStream gos = new GZIPOutputStream(os);
        gos.write(string.getBytes("UTF-8"));
        gos.close();
        byte[] compressed = os.toByteArray();
        os.close();
        return compressed;
    }

    @Nullable
    public static String encodeUrl(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() == null? "Couldn't encode URL in UTF-8": e.getMessage(), e);
        }
        return null;
    }

    @Nullable
    public static String md5Hash(@Nullable String data) throws UnsupportedEncodingException, OutOfMemoryError {
        if (data == null) {
            return null;
        }
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("MD5");
            messageDigest.update(data.getBytes("UTF-8"), 0, data.length());
            byte[] digest = messageDigest.digest();
            StringBuilder hashBuilder = new StringBuilder();
            for (byte b : digest) {
                hashBuilder.append(String.format("%02x", b & 0xff));
            }
            return hashBuilder.toString();
        } catch (NoSuchAlgorithmException e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't hash data", e);
            }
        }
        return null;
    }

    @Nullable
    public static String encodeBase64(String string) throws OutOfMemoryError {
        try {
            byte[] data = string.getBytes("UTF-8");
            return Base64.encodeToString(data, Base64.NO_WRAP);
        } catch (Exception e) {
            if (e.getMessage() != null) {
                InstabugSDKLogger.e(Constants.LOG_TAG, e.getMessage() == null ? "Couldn't encode Base64" : e.getMessage(), e);
            }
        }
        return null;
    }

   public static String encodeHMAC(String key, @NonNull String data) throws Exception {
        Charset asciiCs = Charset.forName("US-ASCII");
        Mac sha256_HMAC = Mac.getInstance("HmacSHA256");
        SecretKeySpec secret_key = new SecretKeySpec(asciiCs.encode(key).array(), "HmacSHA256");
        sha256_HMAC.init(secret_key);
        return Base64.encodeToString(sha256_HMAC.doFinal(asciiCs.encode(data).array()), Base64.NO_WRAP);
    }

    @Nullable
    static String md5Hash(byte[] data) throws UnsupportedEncodingException {
        if (data == null || data.length == 0)
            return null;
        try {
            MessageDigest m = MessageDigest.getInstance("MD5");
            m.update(data, 0, data.length);
            //noinspection UnnecessaryLocalVariable
            String hash = new BigInteger(1, m.digest()).toString(16);
            return hash;
        } catch (NoSuchAlgorithmException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error while generating md5 hash", e);
        }
        return null;
    }
}
