package com.instabug.library.logging;

import com.instabug.library.Constants;
import com.instabug.library.apichecker.APIChecker;
import com.instabug.library.apichecker.VoidRunnable;
import com.instabug.library.model.NetworkLog;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.util.List;
import java.util.Map;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;


/**
 * Created by mNagy on 12/19/16.
 */
public class InstabugNetworkLog {

    @VisibleForTesting
    NetworkLog networkLog;

    public InstabugNetworkLog() {
        networkLog = new NetworkLog();
    }

    /**
     * Extracts HTTP connection properties. Request method, Headers, Date, Url and Response code
     *
     * @param url          Url of the request
     * @param method       request method
     * @param requestBody  the body of the POST request
     * @param responseBody the body of the response
     * @param responseCode the response code
     * @throws IOException
     */
    @Deprecated
    public void log(final String url, final String method, @Nullable final String requestBody, @Nullable final String responseBody, final int responseCode)
            throws IOException {
        APIChecker.checkAndRunInExecutor("NetworkLog.log", new VoidRunnable() {
            @Override
            public void run() throws Exception {
                if (networkLog == null) {
                    networkLog = new NetworkLog();
                }
                networkLog.setResponseCode(responseCode);
                networkLog.setDate(InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds() + "");
                networkLog.setMethod(method);
                networkLog.setUrl(url);

                try {
                    networkLog.setRequest(requestBody);
                    networkLog.setResponse(responseBody);
                } catch (IllegalArgumentException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Content-type is not allowed to be logged");
                    networkLog = null;
                    return;
                }

                insert();
            }
        });
    }

    private void insert() {
        if (networkLog != null) {
            networkLog.insert();
        }
    }

    /**
     * Extracts HTTP connection properties. Request method, Headers, Date, Url and Response code
     *
     * @param connection   instance of the {@link HttpURLConnection}
     * @param requestBody  the body of the POST request
     * @param responseBody the body of the response
     * @throws IOException
     */
    public void log(final HttpURLConnection connection, @Nullable final String requestBody, @Nullable final String responseBody) throws IOException {
        APIChecker.checkAndRunInExecutor("NetworkLog.log", new VoidRunnable() {
            @Override
            public void run() throws Exception {

                if (networkLog == null) {
                    networkLog = new NetworkLog();
                }
                networkLog.setResponseCode(connection.getResponseCode());
                networkLog.setDate(InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds() + "");
                networkLog.setMethod(connection.getRequestMethod());
                networkLog.setUrl(connection.getURL().toString());

                try {
                    addHeaders(connection);
                    networkLog.setRequest(requestBody);
                    networkLog.setResponse(responseBody);
                } catch (JSONException e) {
                    e.printStackTrace();
                } catch (IllegalArgumentException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Content-type is not allowed to be logged");
                    return;
                }

                insert();

            }
        });
    }

    @VisibleForTesting
    void addHeaders(HttpURLConnection connection) throws JSONException, IllegalArgumentException {

        JSONObject jsonObject = new JSONObject();

        //Response headers
        for (String header : connection.getHeaderFields().keySet()) {
            if (header != null) {
                List<String> headerFields = connection.getHeaderFields().get(header);
                if (headerFields != null && !headerFields.isEmpty()) {
                    for (String value : headerFields) {
                        if (header.equalsIgnoreCase(NetworkLog.CONTENT_TYPE)) {
                            if (!value.contains(NetworkLog.JSON)
                                    && !value.contains(NetworkLog.XML_1)
                                    && !value.contains(NetworkLog.XML_2)
                                    && !value.contains(NetworkLog.PROTOBUF)
                                    && !value.contains(NetworkLog.HTML)
                                    && !value.contains(NetworkLog.PLAIN_TEXT)) {
                                throw new IllegalArgumentException();
                            }
                        }
                        jsonObject.put(header, value);
                    }
                }
            }
        }
        if (networkLog == null) {
            networkLog = new NetworkLog();
        }
        networkLog.setRequestHeaders(jsonObject.toString());
        //TODO: find a way to log response headers in manual logging
    }

    /**
     * Extracts HTTP connection properties. Request method, Headers, Date, Url and Response code
     *
     * @param url           Url of the request
     * @param method        request method
     * @param requestBody   the body of the POST request
     * @param responseBody  the body of the response
     * @param responseCode  the response code
     * @param totalDuration total request duration in millis calculated by the user
     */
    public void log(final String url, final String method, @Nullable final String requestBody,
                    @Nullable final String responseBody, final int responseCode, final long totalDuration) {
        APIChecker.checkAndRunInExecutor("NetworkLog.log", new VoidRunnable() {
            @Override
            public void run() {
                if (networkLog == null) {
                    networkLog = new NetworkLog();
                }
                networkLog.setResponseCode(responseCode);
                networkLog.setDate(InstabugDateFormatter.getCurrentUTCTimeStampInMiliSeconds() + "");
                networkLog.setMethod(method);
                networkLog.setUrl(url);
                networkLog.setTotalDuration(totalDuration);

                try {
                    networkLog.setRequest(requestBody);
                    networkLog.setResponse(responseBody);
                } catch (IllegalArgumentException e) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Content-type is not allowed to be logged");
                    networkLog = null;
                    return;
                }

                insert();
            }
        });
    }

    public void log(final String url, final String method, @Nullable final Map<String, String> requestHeaders, @Nullable final String requestBody,
                    Map<String, String> responseHeaders, @Nullable final String responseBody, final int responseCode, final long totalDuration) {

        APIChecker.checkAndRunInExecutor("NetworkLog.log", () -> {
            if (networkLog == null) {
                networkLog = new NetworkLog();
            }

            if (requestHeaders != null) {
                networkLog.setRequestHeaders(new JSONObject(requestHeaders).toString());
            }
            if (responseHeaders != null) {
                networkLog.setResponseHeaders(new JSONObject(responseHeaders).toString());
            }
            log(url, method, requestBody, responseBody, responseCode, totalDuration);
        });
    }
}
