package com.instabug.library.internal.storage.cache;

import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * @author mSobhy
 */
public abstract class Cache<K, V> {
    /**
     * List of listeners that are notified of any changes to this Cache object
     */
    private final List<CacheChangedListener<V>> listeners;

    /**
     * Unique ID which define a cache
     */
    private String id;

    /**
     * Define the app version of the application (to be able to invalidate data from different
     * app version on disk)
     */
    private int appVersion = -1;

    /**
     * Creates a new Cache with ID {@code cacheId}
     *
     * @param cacheId the ID that will be assigned to the current Cache object
     */
    public Cache(String cacheId) {
        this(cacheId, 1);
    }

    /**
     * Creates a new Cache with ID {@code cacheId}
     *
     * @param cacheId    the ID that will be assigned to the current Cache object
     * @param appVersion the app version that this cache will respect (<b>NOT USED YET</b>)
     */
    protected Cache(String cacheId, int appVersion) {
        this.id = cacheId;
        this.appVersion = appVersion;
        this.listeners = new ArrayList<>();
    }

    /**
     * Return the object of the corresponding key from the cache. If no object is available,
     * should return null
     *
     * @param key is the key of the object
     * @return the object of the corresponding key from the cache. If no object is available,
     * should return null
     */
    @Nullable
    public abstract V get(K key);

    /**
     * Puts a value in cache
     *
     * @param key   is the key of the value
     * @param value is the value to put in cache
     * @return should return any existing object with the same key from the cache. If no object
     * is available, should return {@code value}
     */
    public abstract V put(K key, V value);

    /**
     * Delete the corresponding object in cache
     *
     * @param key is the key of the object
     * @return the object that is removed from the cache. If no object is available, should
     * return null
     */
    public abstract V delete(K key);

    /**
     * @return list of all values currently existing in this cache
     */
    public abstract List<V> getValues();

    /**
     * @return the size of the cache (InMemory should return size of Data structure, Disk should
     * return disk size used, etc etc)
     */
    public abstract long size();

    /**
     * Notify all listeners that an item has been removed from this cache
     *
     * @param removedElement element removed from cache
     * @see #listeners
     */
    public void notifyItemRemoved(V removedElement) {
        for (CacheChangedListener<V> cacheChangedListener : listeners) {
            cacheChangedListener.onCachedItemRemoved(removedElement);
        }
    }

    /**
     * Notify all listeners that an item has been added to this cache
     *
     * @param addedElement element added to cache
     * @see #listeners
     */
    public void notifyItemAdded(V addedElement) {
        for (CacheChangedListener<V> cacheChangedListener : listeners) {
            cacheChangedListener.onCachedItemAdded(addedElement);
        }
    }

    /**
     * Notify all listeners that an item has been updated in this cache
     *
     * @param oldElement     old element that has been replaced
     * @param updatedElement new element that replaced the {@code oldElement}
     * @see #listeners
     */
    public void notifyItemUpdated(V oldElement, V updatedElement) {
        for (CacheChangedListener<V> cacheChangedListener : listeners) {
            cacheChangedListener.onCachedItemUpdated(oldElement, updatedElement);
        }
    }

    /**
     * Notify all listeners that this cache's data has been invalidated
     *
     * @see #listeners
     */
    public void notifyCacheInvalidated() {
        for (CacheChangedListener<V> cacheChangedListener : listeners) {
            cacheChangedListener.onCacheInvalidated();
        }
    }

    /**
     * Remove all cached data
     */
    public abstract void invalidate();

    /**
     * @return id of current cache
     */
    public String getId() {
        return id;
    }

    /**
     * @return application version this cache supports
     */
    public int getAppVersion() {
        return appVersion;
    }

    /**
     * Adds {@code CacheChangedListener} to this cache object
     *
     * @param cacheChangedListener Listener to be notified of updates to this cache data
     * @return true if was successfully added, false otherwise
     * @see CacheChangedListener
     */
    public boolean addOnCacheChangedListener(CacheChangedListener<V> cacheChangedListener) {
        return !listeners.contains(cacheChangedListener) && listeners.add(cacheChangedListener);
    }

    /**
     * Removes {@code CacheChangedListener} from this cache object
     *
     * @param cacheChangedListener Listener to be removed from {@link #listeners}
     * @return true if was successfully removed, false otherwise
     * @see CacheChangedListener
     */
    public boolean removeOnCacheChangedListener(CacheChangedListener<V> cacheChangedListener) {
        return listeners.remove(cacheChangedListener);
    }
}
