package com.instabug.library.internal.dataretention;

import com.instabug.library.core.plugin.PluginsManager;
import com.instabug.library.internal.dataretention.core.DataInspector;
import com.instabug.library.internal.dataretention.core.Disposable;
import com.instabug.library.user.UserManager;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

public final class DataGarbageCollector {

    @VisibleForTesting
    final Set<DataInspector<?>> inspectors;

    /**
     * making it package-private will reduce method count since it's accessed from an inner class.
     * Reference: https://academy.realm.io/posts/360andev-jake-wharton-java-hidden-costs-android/
     */
    DataGarbageCollector(Set<DataInspector<?>> inspectors) {
        this.inspectors = inspectors;
    }

    /**
     * Collects disposable data from all plugins
     *
     * @return {@link DisposableData} object containing all the disposable data
     * @deprecated see {@link #collect()}
     */
    @Deprecated
    public static DisposableData deprecatedCollect() {
        List<DisposalPolicy> dataDisposalPolicies = new ArrayList<>();
        dataDisposalPolicies.add(UserManager.getDataDisposalPolicy());
        dataDisposalPolicies.addAll(PluginsManager.getDataDisposalPolicies());
        return new DisposableData(dataDisposalPolicies);
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * Collects all disposables
     *
     * @return {@link Disposable.CompositeDisposable} object containing all the disposable data
     * @see com.instabug.library.internal.dataretention.core.DisposalPolicy
     * @see com.instabug.library.internal.dataretention.core.PolicyCollector
     * @see com.instabug.library.internal.dataretention.core.DataInspector
     */
    @NonNull
    @WorkerThread
    public Disposable collect() {
        Collection<Disposable> disposables = new LinkedList<>();
        for (DataInspector<?> inspector : inspectors) {
            disposables.addAll(inspector.getDisposables());
        }
        return new Disposable.CompositeDisposable(disposables);
    }

    public static class Builder {

        private final Set<DataInspector<?>> inspectors = new HashSet<>();

        Builder() {
        }

        public Builder withInspector(DataInspector<?> inspector) {
            inspectors.add(inspector);
            return this;
        }

        public DataGarbageCollector build() {
            return new DataGarbageCollector(inspectors);
        }
    }
}
