package com.instabug.library.internal.crossplatform

import androidx.annotation.IntDef
import com.instabug.library.networkinterception.NetworkInterceptionServiceLocator
import com.instabug.library.sessionreplay.di.SessionReplayServiceLocator
import com.instabug.library.util.threading.PoolProvider

@IntDef(
    CoreFeature.W3C_EXTERNAL_TRACE_ID,
    CoreFeature.W3C_ATTACHING_GENERATED_HEADER,
    CoreFeature.W3C_ATTACHING_CAPTURED_HEADER
)
@Retention(AnnotationRetention.SOURCE)
annotation class CoreFeature {
    companion object {
        const val W3C_EXTERNAL_TRACE_ID = 0x001
        const val W3C_ATTACHING_GENERATED_HEADER = 0x002
        const val W3C_ATTACHING_CAPTURED_HEADER = 0x003
    }
}

fun interface FeatureStateCallback {
    operator fun invoke(isFeatureEnabled: Boolean)
}

data class CoreFeaturesState(
    val isW3CExternalTraceIdEnabled: Boolean,
    val isAttachingGeneratedHeaderEnabled: Boolean,
    val isAttachingCapturedHeaderEnabled: Boolean,
    val networkLogCharLimit: Int
)

fun interface FeaturesStateListener {
    operator fun invoke(featuresState: CoreFeaturesState)
}

interface ICPConfigurationsProvider {
    val networkLogCharLimit: Int
    fun isFeatureEnabled(@CoreFeature feature: Int, callback: FeatureStateCallback)
    fun isFeatureEnabled(@CoreFeature feature: Int): Boolean
    fun setFeaturesStateListener(listener: FeaturesStateListener?)
    fun onFeaturesStateChanged()
}

object CPConfigurationsProvider : ICPConfigurationsProvider {

    @Volatile
    private var statesListener: FeaturesStateListener? = null

    override val networkLogCharLimit
        get() = SessionReplayServiceLocator.srConfigurationsProvider.networkLogLimit

    override fun isFeatureEnabled(@CoreFeature feature: Int, callback: FeatureStateCallback) {
        PoolProvider.postIOTask { callback(checkFeatureState(feature)) }
    }

    override fun isFeatureEnabled(@CoreFeature feature: Int): Boolean = checkFeatureState(feature)

    override fun setFeaturesStateListener(listener: FeaturesStateListener?) {
        statesListener = listener
    }

    override fun onFeaturesStateChanged() {
        statesListener?.invoke(getFeaturesState())
    }

    private fun checkFeatureState(@CoreFeature feature: Int): Boolean =
        with(NetworkInterceptionServiceLocator.configurationProvider) {
            when (feature) {
                CoreFeature.W3C_EXTERNAL_TRACE_ID -> isW3CNetworkExternalTraceIdEnabled()
                CoreFeature.W3C_ATTACHING_GENERATED_HEADER -> isAttachingGeneratedW3CExternalTraceIdEnabled
                CoreFeature.W3C_ATTACHING_CAPTURED_HEADER -> isAttachingCapturedW3CExternalTraceIdEnabled
                else -> false
            }
        }

    private fun getFeaturesState(): CoreFeaturesState =
        with(NetworkInterceptionServiceLocator.configurationProvider) {
            CoreFeaturesState(
                isW3CNetworkExternalTraceIdEnabled(),
                isAttachingGeneratedW3CExternalTraceIdEnabled,
                isAttachingCapturedW3CExternalTraceIdEnabled,
                networkLogCharLimit = networkLogCharLimit
            )
        }
}
