package com.instabug.library.diagnostics

import com.instabug.library.IBGFeature.Companion.NON_FATAL_ERRORS
import com.instabug.library.Feature.State.ENABLED
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.AppTokenChanged
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.FeaturesFetched
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.OSVersionChanged
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.SdkVersionChanged
import com.instabug.library.diagnostics.configuration.ConfigurationChangedHandler
import com.instabug.library.diagnostics.customtraces.CustomTracesManager
import com.instabug.library.diagnostics.customtraces.di.CustomTracesServiceLocator
import com.instabug.library.diagnostics.customtraces.settings.CustomTracesSettingsResolver
import com.instabug.library.diagnostics.nonfatals.NonFatalsManager
import com.instabug.library.diagnostics.nonfatals.settings.DefaultValues.IS_NON_FATAL_FEATURE_AVAILABLE
import com.instabug.library.diagnostics.sdkEvents.SDKEventsManager
import com.instabug.library.diagnostics.sdkEvents.configurations.SDKEventConfigs
import com.instabug.library.diagnostics.sdkEvents.di.SDKEventsServiceLocator
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator as nonFatalsServiceLocator

/*
 * make changed at diagnostics based on PassedCoreSDKEvents
 */
class DiagnosticsCoreEventHandler {

    private val nonFatalsManager: NonFatalsManager?
        get() = nonFatalsServiceLocator.getNonFatalsManager()
    private val settingsManager: SettingsManager
        get() = SettingsManager.getInstance()
    private val sdkEventsManager: SDKEventsManager
        get() = SDKEventsServiceLocator.sdkEventsManager
    private val sdkEventsConfigs: SDKEventConfigs
        get() = SDKEventsServiceLocator.sdkEventConfigs
    private val configChangedHandlers: List<ConfigurationChangedHandler> = listOf(
        nonFatalsServiceLocator.getNonFatalsConfigurationHandler(),
        SDKEventsServiceLocator.sdkEventsConfigurationChangedHandler,
        CustomTracesServiceLocator.getTracesConfigurationHandler()
    )
    private val customTracesManager: CustomTracesManager =
        CustomTracesServiceLocator.getCustomTracesManager()
    private val isNonFatalEnabled
        get() = SettingsManager.getInstance()
            .getFeatureState(NON_FATAL_ERRORS, IS_NON_FATAL_FEATURE_AVAILABLE) == ENABLED

    /*
    * on sdk version changed all diagnostics databases are cleared
    * on app token changed executionTraces and SDKEvents tables are cleared
    * on os version changed executionTraces and SDKEvents tables are cleared
    * on features request result the diagnostics object is captured to extract sync interval and its subfeatures
    */
    fun handleCoreEvents(event: IBGSdkCoreEvent) {
        when (event) {
            SdkVersionChanged, AppTokenChanged, OSVersionChanged -> clearDiagnosticsCache()
            is FeaturesFetched -> handleFeaturesFetched(event.response)
            is Features -> handleFeatureUpdated(event)
            else -> {}
        }
    }

    private fun clearDiagnosticsCache() {
        nonFatalsManager?.clearCache()
        customTracesManager.clearCache()
        sdkEventsManager.clearCache()
    }

    private fun handleFeatureUpdated(event: Features) {
        nonFatalsManager
            ?.takeUnless { isNonFatalEnabled }
            ?.clearCache()

        if (event == Features.Updated)
            sdkEventsManager
                .takeUnless { sdkEventsConfigs.isEnabled }
                ?.clearCache()
    }


    private fun handleFeaturesFetched(json: String) = runOrLogError {
        json.getDiagnosticsObject()?.apply {
            settingsManager.diagnosticsSyncInterval = optInt(
                DIAGNOSTICS_SYNC_INTERVAL_KEY, DEFAULT_SYNC_INTERVAL
            )
            configChangedHandlers.forEach { handler -> handler.onConfigurationChanged(this) }
        } ?: handleDiagnosticsMissingFromFeatures()
    }.onFailure {
        IBGDiagnostics.reportNonFatalAndLog(
            it,
            "Error in parsing Diagnostics",
            com.instabug.library.Constants.LOG_TAG
        )
    }

    private fun handleDiagnosticsMissingFromFeatures() {
        nonFatalsManager?.clearCache()
        customTracesManager.clearCache()
        CustomTracesSettingsResolver.resetCustomTracesSettings()
    }

    private fun String.getDiagnosticsObject() = JSONObject(this).optJSONObject(DIAGNOSTICS_KEY)

    companion object {
        const val DIAGNOSTICS_KEY = "diagnostics"
        const val DEFAULT_SYNC_INTERVAL: Int = 1440
        const val DIAGNOSTICS_SYNC_INTERVAL_KEY = "sync_interval"
    }
}