package com.instabug.library.visualusersteps

import android.graphics.Bitmap
import android.view.View
import androidx.annotation.VisibleForTesting
import androidx.annotation.WorkerThread
import com.instabug.library.SpansCacheDirectory
import com.instabug.library.interactionstracking.IBGUINode
import com.instabug.library.screenshot.ScreenshotCaptor
import com.instabug.library.screenshot.instacapture.ScreenshotRequest
import com.instabug.library.util.threading.OrderedExecutorService
import java.lang.ref.WeakReference
import java.util.concurrent.Future

class ReproCapturingNotAuthorizedException(message: String) : RuntimeException(message)

private fun notAuthorized(message: String) = ReproCapturingNotAuthorizedException(message)

/**
 * A proxy contract for repro features.
 * Should be implemented as part of delegation, forming a mutual auth proxy with interested
 * parties (features) authorization capabilities to perform its original job.
 */
interface ReproCapturingProxy {
    /**
     * Whether this proxy is authorized or not
     */
    val isAuthorized: Boolean

    /**
     * Evaluate a [ReproConfigurationsProvider].
     * [ReproConfigurationsProvider] is the contract representing repro features state for interested
     * parties (features). When evaluating, the proxy of a specific repro feature will be depending on
     * a defined flag in [ReproConfigurationsProvider], being enabled authorizes the proxy & disabled revokes
     * the previous evaluations auth.
     * @param configProvider the [ReproConfigurationsProvider] that should be used in evaluating the auth state
     * of a given interested party
     */
    fun evaluate(configProvider: ReproConfigurationsProvider)
}

abstract class AbstractReproCapturingProxy(
    private val executor: OrderedExecutorService,
    private val execQueueId: String
) : ReproCapturingProxy {
    private val authorizations: MutableSet<Int> = mutableSetOf()

    protected val internalIsAuthorized: Boolean
        get() = authorizations.isNotEmpty()

    override val isAuthorized: Boolean
        @WorkerThread get() = submitAndGet { internalIsAuthorized }

    /**
     * A functional evaluator, given an object of [ReproConfigurationsProvider] it should contain
     * the specifics of how a certain proxy depends on [ReproConfigurationsProvider] flags.
     */
    protected abstract val evaluator: (ReproConfigurationsProvider) -> Boolean

    override fun evaluate(configProvider: ReproConfigurationsProvider) = execute {
        with(configProvider) {
            if (evaluator(this)) authorize(reproProxyAuthId) else revoke(reproProxyAuthId)
        }
        postEvaluate()
    }

    protected open fun postEvaluate() {
        // No-Op
    }

    private fun execute(job: () -> Unit) {
        executor.execute(execQueueId, job)
    }

    private fun <V> submitAndGet(job: () -> V): V =
        executor.submit(execQueueId, job).get()

    private fun authorize(authId: Int) {
        if (authorizations.contains(authId)) return
        authorizations.add(authId)
    }

    private fun revoke(authId: Int) {
        if (!authorizations.contains(authId)) return
        authorizations.remove(authId)
    }
}

interface ReproScreenshotsCapturingProxy : ReproCapturingProxy, ScreenshotCaptor

@VisibleForTesting
const val ERROR_REPRO_SCREENSHOTS_UNAUTHORIZED =
    "Repro screenshots capturing is disabled for all report types or feature not available"

private const val SCREENSHOT_CAPTURING_QUEUE_ID = "repro-screenshots-exec"

class BasicReproScreenshotsCapturingProxy(
    private val originalCaptor: ScreenshotCaptor,
    private val savingDirectory: SpansCacheDirectory,
    executor: OrderedExecutorService
) : AbstractReproCapturingProxy(executor, SCREENSHOT_CAPTURING_QUEUE_ID),
    ReproScreenshotsCapturingProxy {

    override val evaluator: (ReproConfigurationsProvider) -> Boolean
        get() = { provider -> provider.isReproScreenshotsEnabled }

    override fun capture(request: ScreenshotRequest) {
        if (!internalIsAuthorized) {
            request.listener.onCapturingFailure(notAuthorized(ERROR_REPRO_SCREENSHOTS_UNAUTHORIZED))
            return
        }
        originalCaptor.capture(request)
    }

    override fun postEvaluate() {
        cleanseIfNeeded()
    }

    private fun cleanseIfNeeded() {
        if (internalIsAuthorized) return
        runCatching {
            savingDirectory.currentSpanDirectory
                ?.takeIf { dir -> dir.exists() }
                ?.deleteRecursively()
        }
    }
}

interface ReproStepsCapturingProxy : ReproCapturingProxy, ReproStepsCaptor

private const val STEPS_CAPTURING_QUEUE_ID = "repro-steps-exec"

class BasicReproStepsCapturingProxy(
    private val originalCaptor: ReproStepsCaptor,
    executor: OrderedExecutorService
) : AbstractReproCapturingProxy(executor, STEPS_CAPTURING_QUEUE_ID), ReproStepsCapturingProxy {

    override val evaluator: (ReproConfigurationsProvider) -> Boolean
        get() = { provider -> provider.isReproStepsEnabled }

    override fun postEvaluate() {
        cleanAndResetIfNeeded()
    }

    override fun setLastView(viewRef: WeakReference<View>?) {
        if (!internalIsAuthorized) return
        originalCaptor.setLastView(viewRef)
    }

    override fun getCurrentParent(): Parent? = originalCaptor.getCurrentParent()

    override fun addVisualUserStep(screenName: String, bitmap: Bitmap?) {
        if (!internalIsAuthorized) return
        originalCaptor.addVisualUserStep(screenName, bitmap)
    }

    override fun addVisualUserStep(
        stepType: String,
        screenName: String?,
        view: String?,
        icon: String?
    ) {
        if (!internalIsAuthorized) return
        originalCaptor.addVisualUserStep(stepType, screenName, view, icon)
    }


    override fun addVisualUserStep(
        stepType: String,
        screenName: String?,
        finalTarget: IBGUINode,
        touchedView: Future<TouchedView?>
    ) {
        if (!internalIsAuthorized) return

        originalCaptor.addVisualUserStep(stepType, screenName, finalTarget, touchedView)
    }

    override fun logKeyboardEvent(isKeyboardOpen: Boolean) {
        if (!internalIsAuthorized) return
        originalCaptor.logKeyboardEvent(isKeyboardOpen)
    }

    override fun logFocusChange(oldFocus: View?, newFocus: View?) {
        if (!internalIsAuthorized) return
        originalCaptor.logFocusChange(oldFocus, newFocus)
    }

    override fun logInstabugEnabledStep() {
        if (!internalIsAuthorized) return
        originalCaptor.logInstabugEnabledStep()
    }

    override fun logForegroundStep() {
        if (!internalIsAuthorized) return
        originalCaptor.logForegroundStep()
    }

    override fun logBackgroundStep() {
        if (!internalIsAuthorized) return
        originalCaptor.logBackgroundStep()
    }

    override fun removeLastTapStep() {
        if (!internalIsAuthorized) return
        originalCaptor.removeLastTapStep()
    }

    override fun removeScreenshotId(screenshotUri: String) {
        if (!internalIsAuthorized) return
        originalCaptor.removeScreenshotId(screenshotUri)
    }

    override fun fetch(): ArrayList<VisualUserStep?> {
        if (!internalIsAuthorized) return ArrayList()
        return originalCaptor.fetch()
    }

    override fun clean() {
        originalCaptor.clean()
    }

    override fun reset() {
        originalCaptor.reset()
    }

    override fun duplicateCurrentParent() {
        originalCaptor.duplicateCurrentParent()
    }

    private fun cleanAndResetIfNeeded() {
        if (internalIsAuthorized) return
        with(originalCaptor) { clean(); reset() }
    }
}

class CompositeReproCapturingProxy(
    private val proxiesList: List<ReproCapturingProxy>
) : ReproCapturingProxy {
    override val isAuthorized: Boolean
        @WorkerThread get() = proxiesList.fold(true) { acc, proxy -> acc && proxy.isAuthorized }

    override fun evaluate(configProvider: ReproConfigurationsProvider) {
        proxiesList.forEach { proxy -> proxy.evaluate(configProvider) }
    }
}
